//
//  parsePlotCode.m
//  PopBio
//
//  Created by Stanislav Kounitski on 11/25/07.
//  This work is provided under the terms of the Educational Community License 1.0, a copy of which is included with the source code.
//

#import "parsePlotCode.h"
#import "NSColorAdditions.h"

void parseData(NSArray *results, int *lineIndex, NSMutableDictionary *dict);
NSMutableDictionary *parseDictionary(NSArray *results, int *lineIndex, NSMutableDictionary *rootDict);
NSMutableArray *parseList(NSArray *results, int *lineIndex, NSMutableDictionary *rootDict);
NSMutableArray *parseMatrix(NSArray *results, int *lineIndex);
NSString *parseVerbatim(NSArray *results, int *lineIndex);
void parseLine(NSArray *results, int *lineIndex, NSMutableDictionary *dict, NSMutableDictionary *rootDict);

void parseData(NSArray *results, int *lineIndex, NSMutableDictionary *dict)
{
	int i = *lineIndex;
	
	// read the names of the columns
	NSString *currentLine = results[i];
	NSArray *columnNames = [[NSArray alloc] initWithArray:[currentLine componentsSeparatedByString:@"\t"]];
	int j, numColumns = [columnNames count];
	
	// create a dictionary entry for each column
	for(j=0; j<numColumns; j++) {
		NSMutableArray *column = [[NSMutableArray alloc] init];
		dict[columnNames[j]] = column;
	}
	
	// set currentLine to be the first line of the actual data
	currentLine = results[++i];
	
	// read the data until END_DATA is encountered
	while(![currentLine isEqualToString:@"END DATA"]) {
		
		NSArray *columnEntries = [currentLine componentsSeparatedByString:@"\t"];
		
		for(j=0; j<numColumns; j++) {
			NSMutableArray *column = dict[columnNames[j]];
			double value = [columnEntries[j] doubleValue];
			[column addObject:@(value)];
		}
	
		currentLine = results[++i];
	}
	
	*lineIndex = i;
}

id parseObject(NSArray *results, int *lineIndex, NSString *objectString, NSMutableDictionary *rootDict)
{
	if([objectString isEqualToString:@"MATRIX"])
		return parseMatrix(results, lineIndex);
	if([objectString isEqualToString:@"VERBATIM"])
		return parseVerbatim(results, lineIndex);
	if([objectString isEqualToString:@"LIST"])
		return parseList(results, lineIndex, rootDict);
	if([objectString isEqualToString:@"DICTIONARY"])
		return parseDictionary(results, lineIndex, rootDict);
		
	if([objectString length] > 4) {
		NSString *objectID = [objectString substringWithRange:NSMakeRange(0,3)];
		
		if([objectID isEqualToString:@"CLR"])
			return [NSColor colorWithString:[objectString substringFromIndex:4]];
		
		if([objectID isEqualToString:@"INT"])
			return @([[objectString substringFromIndex:4] intValue]);
		
		if([objectID isEqualToString:@"DBL"])
			return @([[objectString substringFromIndex:4] doubleValue]);
		
		if([objectID isEqualToString:@"REF"]) {
			NSArray *refs = [[objectString substringFromIndex:4] componentsSeparatedByString:@"->"];
			
			id referencedObject = rootDict;
			int i;
			for(i=0; i<[refs count]; i++)
				referencedObject = referencedObject[refs[i]];
				
			if(referencedObject)
				return referencedObject;
				
			return @"";
		}
	}
	
	return objectString;
}

NSMutableDictionary *parseDictionary(NSArray *results, int *lineIndex, NSMutableDictionary *rootDict)
{
	NSMutableDictionary *newDict = [[NSMutableDictionary alloc] init];
	int i = *lineIndex;
	NSString *currentLine = results[i];
	
	// read the data until END DICTIONARY is encountered
	while(![currentLine isEqualToString:@"END DICTIONARY"])
	{
		parseLine(results, &i, newDict, rootDict);
		currentLine = results[i];
	}
	
	*lineIndex = i+1;
	return newDict;
}

NSMutableArray *parseList(NSArray *results, int *lineIndex, NSMutableDictionary *rootDict)
{
	NSMutableArray *list = [[NSMutableArray alloc] init];
	NSString *currentLine = results[(*lineIndex)];
	
	// read the data until END LIST is encountered
	while(![currentLine isEqualToString:@"END LIST"])
	{
		(*lineIndex)++;
		[list addObject:parseObject(results, lineIndex, currentLine, rootDict)];
		currentLine = results[(*lineIndex)];
	}
	
	(*lineIndex)++;
	return list;
}

NSMutableArray *parseMatrix(NSArray *results, int *lineIndex)
{
	NSMutableArray *matrix = [[NSMutableArray alloc] init];
	int i = *lineIndex;
	NSString *currentLine = results[i];
	
	// read the data until END MATRIX is encountered
	while(![currentLine isEqualToString:@"END MATRIX"])
	{
		[matrix addObject:[[NSMutableArray alloc]
			initWithArray:[currentLine componentsSeparatedByString:@"\t"]]];
		currentLine = results[++i];
	}
	
	*lineIndex = i+1;
	return matrix;
}

NSString *parseVerbatim(NSArray *results, int *lineIndex)
{
	int lastLine, i, lineLength=0;
	for(lastLine = *lineIndex; ![results[lastLine] isEqualToString:@"END VERBATIM"]; lastLine++);
	
	for(i = *lineIndex; i<lastLine; i++)
		lineLength += [results[i] length];
	
	NSMutableString *verbatimStr = [NSMutableString stringWithCapacity:lineLength];
	
	[verbatimStr appendString:results[(*lineIndex)]];
	for(i = *lineIndex+1; i<lastLine; i++)
		[verbatimStr appendFormat:@"\n%@", results[i]];
	
	(*lineIndex) = lastLine+1;
	return verbatimStr;
}

void parseLine(NSArray *results, int *lineIndex, NSMutableDictionary *dict, NSMutableDictionary *rootDict)
{
	NSString *line = results[(*lineIndex)];
	(*lineIndex)++;
	
	if([line isEqualToString:@""])
		return;
	
	if([line isEqualToString:@"BEGIN DATA"]) {
		parseData(results, lineIndex, dict);
		return;
	}

	NSArray *subStrings = [line componentsSeparatedByString:@": "];
	if([subStrings count] != 2) {
//		NSAssert(0, [[NSString stringWithFormat:
//			@"Parser error on line %d: unable to parse \"%@\".", *lineIndex-1, [str cString]] cString]);
		return;
	}
	
	dict[subStrings[0]] = parseObject(results, lineIndex, subStrings[1], rootDict);
}

void parsePlotCode(NSString* str, NSMutableDictionary *rootDict)
{
	NSMutableArray *results = [[NSMutableArray alloc] initWithArray:[str componentsSeparatedByString:@"\n"]];

	// remove any blank lines at the end of the output
	NSString *lastLine = [results lastObject];
	while([lastLine isEqualToString:@""]) {
		[results removeLastObject];
		lastLine = [results lastObject];
	}
	
	// remove "EXPERIMENT_DONE", if it is the last line
	if([[results lastObject] isEqualToString:@"EXPERIMENT_DONE"])
		[results removeLastObject];
	
	int i=0, numLines = [results count];
	
	while(i<numLines)
		parseLine(results, &i, rootDict, rootDict);
}