//
//  PredatorPreyDocument.m
//  PopBio
//
//  Created by Aaron Golden on 05/09/07.
//  Copyright Reed College 2007 . All rights reserved.
//

#import "PredatorPreyDocument.h"
#import "PlotWindowController.h"
#import "PopBioController.h"
#import "FineTunePanel.h"
#import "NSColorAdditions.h"

@implementation PredatorPreyDocument

// ----------------
// init and dealloc
// ----------------

- (id)init
{
	self = [super init];
	if(self){
		NSString *str = [[NSBundle bundleForClass:[self class]] pathForImageResource:@"targetCursor.png"];
		_crosshairImage = [[NSImage alloc] initWithContentsOfFile:str];
		
		currentModel = ModelTypeLotkaVolterra;
		
		LVParams.modelType = ModelTypeLotkaVolterra;
		LVParams.timeIntervals = 10.0;
		LVParams.initialPreyPopulation = 20.0;
		LVParams.initialPredatorPopulation = 4.0;
		LVParams.preyBirthRate = 1.0;
		LVParams.predatorDeathRate = 0.5;
		LVParams.preyCaptureProbability = 0.3;
		LVParams.predatorEnergyEfficiency = 0.02;
		LVParams.criticalPreyDensity = 0;
		LVParams.preyCarryingCapacity = 0;
		LVParams.maxPredatorAttackRate = 0;
		LVParams.halfAttackRateDensity = 0;
		LVParams.integratorType = IntegratorTypeRungeKutta;
		LVParams.pointStep = 0.1;
		LVParams.randomSeed = time(0);
		
		LesParams.modelType = ModelTypeLeslie;
		LesParams.timeIntervals = 100.0;
		LesParams.initialPreyPopulation = 20.0;
		LesParams.initialPredatorPopulation = 4.0;
		LesParams.preyBirthRate = 1.0;
		LesParams.predatorDeathRate = 0.3;
		LesParams.preyCaptureProbability = 0.5;
		LesParams.predatorEnergyEfficiency = 0;
		LesParams.criticalPreyDensity = 4;
		LesParams.preyCarryingCapacity = 100;
		LesParams.maxPredatorAttackRate = 0;
		LesParams.halfAttackRateDensity = 0;
		LesParams.integratorType = IntegratorTypeRungeKutta;
		LesParams.pointStep = 0.1;
		LesParams.randomSeed = time(0);
		
		H1Params.modelType = ModelTypeHollingI;
		H1Params.timeIntervals = 2000.0;
		H1Params.initialPreyPopulation = 20.0;
		H1Params.initialPredatorPopulation = 4.0;
		H1Params.preyBirthRate = 0.1;
		H1Params.predatorDeathRate = 0.05;
		H1Params.preyCaptureProbability = 0.3;
		H1Params.predatorEnergyEfficiency = 0.02;
		H1Params.criticalPreyDensity = 0;
		H1Params.preyCarryingCapacity = 100;
		H1Params.maxPredatorAttackRate = 0.5;
		H1Params.halfAttackRateDensity = 70;
		H1Params.integratorType = IntegratorTypeRungeKutta;
		H1Params.pointStep = 5.0;
		H1Params.randomSeed = time(0);
		
		H2Params.modelType = ModelTypeHollingII;
		H2Params.timeIntervals = 2000.0;
		H2Params.initialPreyPopulation = 20.0;
		H2Params.initialPredatorPopulation = 4.0;
		H2Params.preyBirthRate = 0.2;
		H2Params.predatorDeathRate = 0.04;
		H2Params.preyCaptureProbability = 0.7;
		H2Params.predatorEnergyEfficiency = 0.01;
		H2Params.criticalPreyDensity = 0;
		H2Params.preyCarryingCapacity = 100;
		H2Params.maxPredatorAttackRate = 0.3;
		H2Params.halfAttackRateDensity = 1;
		H2Params.integratorType = IntegratorTypeRungeKutta;
		H2Params.pointStep = 5.0;
		H2Params.randomSeed = time(0);
	}
	return self;
}


// --------------------------------------------------
// implementation of the document-specific parameters
// --------------------------------------------------

- (void)synchronizeParamsToUserInterface
{
	currentModel = [[[tabView selectedTabViewItem] identifier] intValue];

	LVParams.initialPreyPopulation = [initialPreyPopulationTextField doubleValue];
	LVParams.initialPredatorPopulation = [initialPredatorPopulationTextField doubleValue];
	LVParams.timeIntervals = [LVTimeIntervalsTextField doubleValue];
	LVParams.preyBirthRate = [LVPreyBirthRateTextField doubleValue];
	LVParams.predatorDeathRate = [LVPredatorDeathRateTextField doubleValue];
	LVParams.predatorEnergyEfficiency = [LVPredatorEnergyEfficiencyTextField doubleValue];
	LVParams.preyCaptureProbability = [LVPreyCaptureProbabilityTextField doubleValue];
	LVParams.integratorType = [LVFineTunePanel integratorType];
	LVParams.pointStep = [LVFineTunePanel pointStep];
	
	LesParams.initialPreyPopulation = [initialPreyPopulationTextField doubleValue];
	LesParams.initialPredatorPopulation = [initialPredatorPopulationTextField doubleValue];
	LesParams.timeIntervals = [LesTimeIntervalsTextField doubleValue];
	LesParams.preyBirthRate = [LesPreyBirthRateTextField doubleValue];
	LesParams.predatorDeathRate = [LesPredatorDeathRateTextField doubleValue];
	LesParams.preyCaptureProbability = [LesPreyCaptureProbabilityTextField doubleValue];
	LesParams.criticalPreyDensity = [LesCriticalPreyDensityTextField doubleValue];
	LesParams.preyCarryingCapacity = [LesPreyCarryingCapacityTextField doubleValue];
	LesParams.integratorType = [LesFineTunePanel integratorType];
	LesParams.pointStep = [LesFineTunePanel pointStep];
	
	H1Params.initialPreyPopulation = [initialPreyPopulationTextField doubleValue];
	H1Params.initialPredatorPopulation = [initialPredatorPopulationTextField doubleValue];
	H1Params.timeIntervals = [H1TimeIntervalsTextField doubleValue];
	H1Params.preyBirthRate = [H1PreyBirthRateTextField doubleValue];
	H1Params.predatorDeathRate = [H1PredatorDeathRateTextField doubleValue];
	H1Params.predatorEnergyEfficiency = [H1PredatorEnergyEfficiencyTextField doubleValue];
	H1Params.preyCaptureProbability = [H1PreyCaptureProbabilityTextField doubleValue];
	H1Params.preyCarryingCapacity = [H1PreyCarryingCapacityTextField doubleValue];
	H1Params.maxPredatorAttackRate = [H1MaxPredatorAttackRateTextField doubleValue];
	H1Params.halfAttackRateDensity = [H1HalfAttackRateDensityTextField doubleValue];
	H1Params.integratorType = [H1FineTunePanel integratorType];
	H1Params.pointStep = [H1FineTunePanel pointStep];
	
	H2Params.initialPreyPopulation = [initialPreyPopulationTextField doubleValue];
	H2Params.initialPredatorPopulation = [initialPredatorPopulationTextField doubleValue];
	H2Params.timeIntervals = [H2TimeIntervalsTextField doubleValue];
	H2Params.preyBirthRate = [H2PreyBirthRateTextField doubleValue];
	H2Params.predatorDeathRate = [H2PredatorDeathRateTextField doubleValue];
	H2Params.predatorEnergyEfficiency = [H2PredatorEnergyEfficiencyTextField doubleValue];
	H2Params.preyCaptureProbability = [H2PreyCaptureProbabilityTextField doubleValue];
	H2Params.preyCarryingCapacity = [H2PreyCarryingCapacityTextField doubleValue];
	H2Params.maxPredatorAttackRate = [H2MaxPredatorAttackRateTextField doubleValue];
	H2Params.halfAttackRateDensity = [H2HalfAttackRateDensityTextField doubleValue];
	H2Params.integratorType = [H2FineTunePanel integratorType];
	H2Params.pointStep = [H2FineTunePanel pointStep];
}

- (void)synchronizeUserInterfaceToParams
{
	[initialPreyPopulationTextField      setDoubleValue:LVParams.initialPreyPopulation];
	[initialPredatorPopulationTextField  setDoubleValue:LVParams.initialPredatorPopulation];
	
	[LVTimeIntervalsTextField            setDoubleValue:LVParams.timeIntervals];
	[LVPreyBirthRateTextField            setDoubleValue:LVParams.preyBirthRate];
	[LVPredatorDeathRateTextField        setDoubleValue:LVParams.predatorDeathRate];
	[LVPredatorEnergyEfficiencyTextField setDoubleValue:LVParams.predatorEnergyEfficiency];
	[LVPreyCaptureProbabilityTextField   setDoubleValue:LVParams.preyCaptureProbability];
	[LVFineTunePanel                     setIntegratorType:LVParams.integratorType];
	[LVFineTunePanel                     setPointStep:LVParams.pointStep];
	
	[LesTimeIntervalsTextField            setDoubleValue:LesParams.timeIntervals];
	[LesPreyBirthRateTextField            setDoubleValue:LesParams.preyBirthRate];
	[LesPredatorDeathRateTextField        setDoubleValue:LesParams.predatorDeathRate];
	[LesPreyCaptureProbabilityTextField   setDoubleValue:LesParams.preyCaptureProbability];
	[LesCriticalPreyDensityTextField      setDoubleValue:LesParams.criticalPreyDensity];
	[LesPreyCarryingCapacityTextField     setDoubleValue:LesParams.preyCarryingCapacity];
	[LesFineTunePanel                     setIntegratorType:LesParams.integratorType];
	[LesFineTunePanel                     setPointStep:LesParams.pointStep];
	
	[H1TimeIntervalsTextField            setDoubleValue:H1Params.timeIntervals];
	[H1PreyBirthRateTextField            setDoubleValue:H1Params.preyBirthRate];
	[H1PredatorDeathRateTextField        setDoubleValue:H1Params.predatorDeathRate];
	[H1PredatorEnergyEfficiencyTextField setDoubleValue:H1Params.predatorEnergyEfficiency];
	[H1PreyCaptureProbabilityTextField   setDoubleValue:H1Params.preyCaptureProbability];
	[H1PreyCarryingCapacityTextField     setDoubleValue:H1Params.preyCarryingCapacity];
	[H1MaxPredatorAttackRateTextField    setDoubleValue:H1Params.maxPredatorAttackRate];
	[H1HalfAttackRateDensityTextField    setDoubleValue:H1Params.halfAttackRateDensity];
	[H1FineTunePanel                     setIntegratorType:H1Params.integratorType];
	[H1FineTunePanel                     setPointStep:H1Params.pointStep];
	
	[H2TimeIntervalsTextField            setDoubleValue:H2Params.timeIntervals];
	[H2PreyBirthRateTextField            setDoubleValue:H2Params.preyBirthRate];
	[H2PredatorDeathRateTextField        setDoubleValue:H2Params.predatorDeathRate];
	[H2PredatorEnergyEfficiencyTextField setDoubleValue:H2Params.predatorEnergyEfficiency];
	[H2PreyCaptureProbabilityTextField   setDoubleValue:H2Params.preyCaptureProbability];
	[H2PreyCarryingCapacityTextField     setDoubleValue:H2Params.preyCarryingCapacity];
	[H2MaxPredatorAttackRateTextField    setDoubleValue:H2Params.maxPredatorAttackRate];
	[H2HalfAttackRateDensityTextField    setDoubleValue:H2Params.halfAttackRateDensity];
	[H2FineTunePanel                     setIntegratorType:H2Params.integratorType];
	[H2FineTunePanel                     setPointStep:H2Params.pointStep];
	
	[tabView selectTabViewItemAtIndex:currentModel];
	[self updateTabs];
}

- (NSArray*)argumentsForEvaluator
{
	PredatorPreyParameters *allParams[4] = {&LVParams, &LesParams, &H1Params, &H2Params};
	PredatorPreyParameters *params = allParams[currentModel];

	return @[[NSString stringWithFormat:@"%d", kExperimentTypePredatorPrey],
		[NSString stringWithFormat:@"%d", params->modelType],
		[NSString stringWithFormat:@"%lf", params->timeIntervals],
		[NSString stringWithFormat:@"%lf", params->initialPreyPopulation],
		[NSString stringWithFormat:@"%lf", params->initialPredatorPopulation],
		[NSString stringWithFormat:@"%lf", params->preyBirthRate],
		[NSString stringWithFormat:@"%lf", params->predatorDeathRate],
		[NSString stringWithFormat:@"%lf", params->predatorEnergyEfficiency],
		[NSString stringWithFormat:@"%lf", params->preyCaptureProbability],
		[NSString stringWithFormat:@"%lf", params->criticalPreyDensity],
		[NSString stringWithFormat:@"%lf", params->preyCarryingCapacity],
		[NSString stringWithFormat:@"%lf", params->maxPredatorAttackRate],
		[NSString stringWithFormat:@"%lf", params->halfAttackRateDensity],
		[NSString stringWithFormat:@"%.8f", [[[NSUserDefaults standardUserDefaults]
			objectForKey:@"CompetitionStabilityThreshold"] doubleValue]],
		[NSString stringWithFormat:@"%d", params->integratorType],
		[NSString stringWithFormat:@"%lf", params->pointStep],
		[NSString stringWithFormat:@"%d", params->randomSeed++]];
}

- (NSString *)parametersReport;
{
	PredatorPreyParameters *allParams[4] = {&LVParams, &LesParams, &H1Params, &H2Params};
	PredatorPreyParameters *params = allParams[currentModel];

	NSMutableString *str = [NSMutableString string];
	[str appendFormat:@"Predator-Prey:\n\n"];

    switch (currentModel) {
        case ModelTypeLotkaVolterra:
            [str appendFormat:@"Lotka-Volterra Model\n"];
            break;
        case ModelTypeLeslie:
            [str appendFormat:@"Leslie Model\n"];
            break;
        case ModelTypeHollingI:
            [str appendFormat:@"Holling I Model\n"];
            break;
        case ModelTypeHollingII:
            [str appendFormat:@"Holling II Model\n"];
            break;
    }
	
	[str appendFormat:@"Time Intervals: %.2lf\n",params->timeIntervals];
	[str appendFormat:@"Initial Prey Population: %.2lf\n",params->initialPreyPopulation];
	[str appendFormat:@"Initial Predator Population: %.2lf\n",params->initialPredatorPopulation];
	[str appendFormat:@"Prey Birth Rate: %.3lf\n",params->preyBirthRate];
	[str appendFormat:@"Predator Death Rate: %.3lf\n",params->predatorDeathRate];
	[str appendFormat:@"Predator Energy Efficiency: %.4f\n",params->predatorEnergyEfficiency];
	
	if(currentModel == ModelTypeLotkaVolterra) {
		[str appendFormat:@"Prey Capture Probability: %.3lf\n",params->preyCaptureProbability];
	} else if(currentModel == ModelTypeLeslie) {
		[str appendFormat:@"Prey Capture Probability: %.3lf\n",params->preyCaptureProbability];
		[str appendFormat:@"Critical Prey Density: %.3lf\n",params->criticalPreyDensity];
		[str appendFormat:@"Prey Carrying Capacity: %.3lf\n",params->preyCarryingCapacity];
	} else {
		[str appendFormat:@"Prey Carrying Capacity: %.3lf\n",params->preyCarryingCapacity];
		[str appendFormat:@"Maximum Predator Attack Rate: %.3lf\n",params->maxPredatorAttackRate];
		[str appendFormat:@"1/2 Saturation Attack Rate: %.3lf\n",params->halfAttackRateDensity];
	}
	
	[str appendFormat:@"Epsilon: %.8lf\n",[[[NSUserDefaults standardUserDefaults]
		objectForKey:@"CompetitionStabilityThreshold"] doubleValue]];	
	if(params->integratorType == IntegratorTypeRungeKutta)
		[str appendFormat:@"Runge-Kutta Integration\n"];
	else if(params->integratorType == IntegratorTypeEuler)
		[str appendFormat:@"Euler forward Integration\n"];
	[str appendFormat:@"Point step: %lf",params->pointStep];
	return str;
}

// ------------------------------------------------------
// document-specific implementation of setPlotCodeStrings
// ------------------------------------------------------

- (NSMutableString*)isoclineString
{
	PredatorPreyParameters *allParams[4] = {&LVParams, &LesParams, &H1Params, &H2Params};
	PredatorPreyParameters *params = allParams[currentModel];

	NSMutableString *isoStr = [NSMutableString string];
	
	double alpha = params->preyBirthRate;
	double beta = params->preyCaptureProbability;
	
	if(currentModel == ModelTypeLotkaVolterra) {
		double gamma = params->predatorDeathRate;
		double delta = beta * params->predatorEnergyEfficiency;
	
		[generatedPlotCode appendFormat:
			@"Prey Isocline: DICTIONARY\nspecial-curve: line\nslope: INT 0\ny-intercept: DBL %lf\ntitle: Prey Isocline\ncolor: REF Prey Color\ncurve-id: %d\nEND DICTIONARY\n\n",
			alpha/beta, [[PBController nextCurveID] intValue]];
				
		[generatedPlotCode appendFormat:
			@"Predator Isocline: DICTIONARY\nspecial-curve: vLine\nx: DBL %lf\ntitle: Predator Isocline\ncolor: REF Predator Color\ncurve-id: %d\nEND DICTIONARY\n",
			gamma/delta, [[PBController nextCurveID] intValue]];
	}
	else if(currentModel == ModelTypeLeslie) {
		double K = params->preyCarryingCapacity;
		double e = params->criticalPreyDensity;
	
		[generatedPlotCode appendFormat:
			@"Prey Isocline: DICTIONARY\nspecial-curve: line\nslope: DBL %lf\ny-intercept: DBL %lf\ntitle: Prey Isocline\ncolor: REF Prey Color\ncurve-id: %d\nEND DICTIONARY\n\n",
			-alpha/(beta*K), alpha/beta, [[PBController nextCurveID] intValue]];
				
		[generatedPlotCode appendFormat:
			@"Predator Isocline: DICTIONARY\nspecial-curve: line\nslope: DBL %lf\ny-intercept: INT 0\ntitle: Predator Isocline\ncolor: REF Predator Color\ncurve-id: %d\nEND DICTIONARY\n",
			1/e, [[PBController nextCurveID] intValue]];
	}
	else {
		double K = params->preyCarryingCapacity;
		double m = params->maxPredatorAttackRate;
		double w = params->halfAttackRateDensity;
		double gamma = params->predatorDeathRate;
		double delta = beta * params->predatorEnergyEfficiency;
		
		if(currentModel == ModelTypeHollingI) {
			[generatedPlotCode appendFormat:
				@"Prey Isocline: DICTIONARY\nspecial-curve: parabola\na: DBL %lf\nb: DBL %lf\nc: DBL %lf\ntitle: Prey Isocline\ncolor: REF Prey Color\ncurve-id: %d\nEND DICTIONARY\n\n",
				-alpha/(K*m), (alpha/m)*(1-w/K), w*alpha/m, [[PBController nextCurveID] intValue]];
		
			[generatedPlotCode appendFormat:
				@"Predator Isocline: DICTIONARY\nspecial-curve: vLine\nx: DBL %lf\ntitle: Predator Isocline\ncolor: REF Predator Color\ncurve-id: %d\nEND DICTIONARY\n",
				w/(gamma*m/delta - 1), [[PBController nextCurveID] intValue]];
		}
		else if(currentModel == ModelTypeHollingII) {
			[generatedPlotCode appendFormat:
				@"Prey Isocline: DICTIONARY\nspecial-curve: Holling II isocline\nK: DBL %lf\na: DBL %lf\nm: DBL %lf\nw: DBL %lf\ntitle: Prey Isocline\ncolor: REF Prey Color\ncurve-id: %d\nEND DICTIONARY\n\n",
				K, alpha, m, w, [[PBController nextCurveID] intValue]];
		
			[generatedPlotCode appendFormat:
				@"Predator Isocline: DICTIONARY\nspecial-curve: line\nslope: DBL %lf\ny-intercept: INT 0\ntitle: Predator Isocline\ncolor: REF Predator Color\ncurve-id: %d\nEND DICTIONARY\n",
				(gamma*m/delta - 1)/w, [[PBController nextCurveID] intValue]];
		}
	}
	
	return isoStr;
}

- (void)setPlotCodeStrings
{
	// create the fixedPlotCode string, if necessary
	if(!fixedPlotCode)
		fixedPlotCode = [NSString stringWithContentsOfFile:[[NSBundle bundleForClass:[self class]]
			pathForResource:@"predatorPreyPlotCode" ofType:@"txt"]  encoding:NSUTF8StringEncoding error:NULL];
	
	generatedPlotCode = [[NSMutableString alloc] init];
	
	// generate generatedPlotCode
	[generatedPlotCode appendFormat:
		@"Params: VERBATIM\n%@\nEND VERBATIM\n\n",
		[self parametersReport]];
	
	[generatedPlotCode appendFormat:
		@"Prey Color: CLR %@\nPredator Color: CLR %@\n\n",
		[[preyColorWell color] stringValue], [[predatorColorWell color] stringValue]];
	
	[generatedPlotCode appendFormat:
		@"Curve ID 1: INT %d\nCurve ID 2: INT %d\nCurve ID 3: INT %d\nCurve ID 4: INT %d\nCurve ID 5: INT %d\n\n",
		[[PBController nextCurveID] intValue], [[PBController nextCurveID] intValue], [[PBController nextCurveID] intValue], [[PBController nextCurveID] intValue], [[PBController nextCurveID] intValue]];
	
	[generatedPlotCode appendString:[self isoclineString]];
		
	//NSLog(generatedPlotCode);
}

// -----------------------
// miscellaneous functions
// -----------------------

- (NSString*)titleForPlotWindow
{
	return [NSString stringWithFormat:@"Predator-Prey Graphics: %@", [self displayName]];
}

- (void)registerForNotifications
{
//	[[NSNotificationCenter defaultCenter] addObserver:self
//		selector:@selector(controlTextDidEndEditing:)
//		name:@"NSControlTextDidEndEditingNotification" object:nil];
}

- (NSString *)windowNibName
{
    return @"PredatorPreyDocument";
}
- (NSData *)dataOfType:(NSString *)typeName error:(NSError **)outError
{
	[self synchronizeParamsToUserInterface];
	NSMutableData *data = [NSMutableData data];
	[data appendBytes:&currentModel length:sizeof(int)];
	[data appendBytes:&LVParams     length:sizeof(PredatorPreyParameters)];
	[data appendBytes:&LesParams    length:sizeof(PredatorPreyParameters)];
	[data appendBytes:&H1Params     length:sizeof(PredatorPreyParameters)];
	[data appendBytes:&H2Params     length:sizeof(PredatorPreyParameters)];
	return data;
}

- (BOOL)readFromData:(NSData *)data ofType:(NSString *)typeName error:(NSError **)outError
{
	int intSize = sizeof(int);
	int paramSize = sizeof(PredatorPreyParameters);
	
	[data getBytes:&currentModel range:NSMakeRange(0, intSize)];
	[data getBytes:&LVParams  range:NSMakeRange(intSize, paramSize)];
	[data getBytes:&LesParams range:NSMakeRange(intSize+paramSize, paramSize)];
	[data getBytes:&H1Params  range:NSMakeRange(intSize+paramSize*2, paramSize)];
	[data getBytes:&H2Params  range:NSMakeRange(intSize+paramSize*3, paramSize)];
	
	[self synchronizeUserInterfaceToParams];
    return YES;
}

/*- (void)controlTextDidEndEditing:(NSNotification *)aNotification
{
	[self updateInitialPopulationPanelPlotView];
}*/

#define kWindowHeightLotkaVolterra 416.0
#define kWindowHeightLeslie 443.0
#define kWindowHeightHolling 497.0

- (void)updateTabs
{
	[[tabView tabViewItemAtIndex:0] setLabel:(currentModel==0 ? @"Lotka-Volterra" : @"L-V")];
	[[tabView tabViewItemAtIndex:1] setLabel:(currentModel==1 ? @"Leslie" : @"L")];
	[[tabView tabViewItemAtIndex:2] setLabel:(currentModel==2 ? @"Holling I" : @"H1")];
	[[tabView tabViewItemAtIndex:3] setLabel:(currentModel==3 ? @"Holling II" : @"H2")];

	NSWindow *theWindow = [self windowForSheet];
	NSRect frame = [theWindow frame];
	NSSize newSize = frame.size;
	
	if(currentModel == ModelTypeLotkaVolterra)
		newSize.height = kWindowHeightLotkaVolterra;
	else if(currentModel == ModelTypeLeslie)
		newSize.height = kWindowHeightLeslie;
	else
		newSize.height = kWindowHeightHolling;

	frame.origin.y -= newSize.height - frame.size.height;
	frame.size = newSize;
	[theWindow setFrame:frame display:YES animate:YES];
}

- (IBAction)helpButton:(id)sender
{
	[PBController openHelp:self];
}

@end

@implementation PredatorPreyDocument(NSNibAwaking)

-(void)awakeFromNib
{
	[self registerForNotifications];
}
@end

@implementation PredatorPreyDocument (NSTabViewDelegate)

- (void)tabView:(NSTabView *)tabView didSelectTabViewItem:(NSTabViewItem *)tabViewItem
{
	[self synchronizeParametersAndUserInterface:NULL];
}
@end