/*
 *  PredatorPrey.c
 *  PopBio
 *
 *  Created by Aaron Golden on 05/09/07.
 *  This work is provided under the terms of the Educational Community License 1.0, a copy of which is included with the source code.
 *
 */
 
#include <stdio.h>
#include <math.h>
#include <stdlib.h>

#include "Backend.h"
#include "PredatorPrey.h"
#include "RungeKutta.h"

double _alpha, _beta, _gamma, _delta, _preyCarrCap, _critPreyDensity, _dt, _maxPredAttackRate, _halfSaturatedPreyDensity;

pair xPrimeLotkaVolterra(double t, pair x)
{
	pair xP;
	xP.a =  x.a * (_alpha - _beta * x.b);
	xP.b = -x.b * (_gamma - _delta * x.a);
	return xP;
}

pair xPrimeLeslie(double t, pair x)
{
	pair xP;
	
	xP.a = _alpha*x.a*(1.0 - x.a/_preyCarrCap)-_beta*x.a*x.b;
	xP.b = _gamma*x.b*(1.0 - _critPreyDensity*x.b/x.a);
	
	return xP;
}

pair xPrimeHollingI(double t, pair x)
{
	pair xP;
	
	double b = _maxPredAttackRate*x.a/(_halfSaturatedPreyDensity + x.a);
	xP.a = _alpha*x.a*(1.0 - x.a/_preyCarrCap) - b*x.b;
	xP.b = (_gamma*b - _delta)*x.b;
	
	return xP;
}

pair xPrimeHollingII(double t, pair x)
{
	pair xP;
	
	double b = _maxPredAttackRate*x.a/(_halfSaturatedPreyDensity*x.b + x.a);
	xP.a = _alpha*x.a*(1.0 - x.a/_preyCarrCap) - b*x.b;
	xP.b = (_gamma*b - _delta)*x.b;
	
	return xP;
}

int runPredatorPrey(int argc, char *argv[])
{
	if(argc < 18){
		printf("PredatorPrey requires parameters:  <timeIntervals> <initial prey population>\
		<initial predator population> <prey birth rate> <predator death rate>\
		<prey escape probability> <predator energy efficiency>\
		<epsilon> <integrator type> <point step> <random seed>\n");
		return -1;
	}

	PredatorPreyParameters params;
	sscanf(argv[2], "%d",  &params.modelType);
	sscanf(argv[3], "%lf", &params.timeIntervals);
	sscanf(argv[4], "%lf", &params.initialPreyPopulation);
	sscanf(argv[5], "%lf", &params.initialPredatorPopulation);
	sscanf(argv[6], "%lf", &params.preyBirthRate);
	sscanf(argv[7], "%lf", &params.predatorDeathRate);
	sscanf(argv[8], "%lf", &params.predatorEnergyEfficiency);
	sscanf(argv[9], "%lf", &params.preyCaptureProbability);
	sscanf(argv[10], "%lf", &params.criticalPreyDensity);
	sscanf(argv[11], "%lf", &params.preyCarryingCapacity);
	sscanf(argv[12], "%lf", &params.maxPredatorAttackRate);
	sscanf(argv[13], "%lf", &params.halfAttackRateDensity);
	sscanf(argv[14], "%lf", &params.epsilon);
	sscanf(argv[15], "%d", &params.integratorType);
	sscanf(argv[16], "%lf", &params.pointStep);
	sscanf(argv[17], "%d", &params.randomSeed);
	
	srandom(params.randomSeed);
	
	int numTimeSteps = (int)ceil(params.timeIntervals/params.pointStep)+1;
	pair x;
	x.a = params.initialPreyPopulation;
	x.b = params.initialPredatorPopulation;
	
	_alpha = params.preyBirthRate;
	_beta = params.preyCaptureProbability;
	_gamma = params.predatorDeathRate;
	_delta = _beta * params.predatorEnergyEfficiency;
	_preyCarrCap = params.preyCarryingCapacity;
	_critPreyDensity = params.criticalPreyDensity;
	_maxPredAttackRate = params.maxPredatorAttackRate;
	_halfSaturatedPreyDensity = params.halfAttackRateDensity;
	_dt = params.pointStep;
	
	double *x1Table = (double*)malloc(sizeof(double)*numTimeSteps);
	double *x2Table = (double*)malloc(sizeof(double)*numTimeSteps);
	x1Table[0] = x.a;
	x2Table[0] = x.b;

	double t=0.0;
	int j;
	//int stabilized = 0;
	
	printf("\nalpha: %lf\n", _alpha);
	printf("beta: %lf\n", _beta);
	printf("gamma: %lf\n", _gamma);
	printf("delta: %lf\n", _delta);
	
	printf("\nBEGIN DATA\n");
	printf("Time\tPrey\tPredator\n");
	printf("%lf\t\%lf\t\%lf\n", t, x.a, x.b);

	pair (*xPrime)(double, pair);
	switch (params.modelType) {
        case ModelTypeLotkaVolterra:
            xPrime = &xPrimeLotkaVolterra;
            break;
        case ModelTypeLeslie:
            xPrime = &xPrimeLeslie;
            break;
        case ModelTypeHollingI:
            xPrime = &xPrimeHollingI;
            break;
        case ModelTypeHollingII:
            xPrime = &xPrimeHollingII;
            break;
    }
	
	for(j=1; j < numTimeSteps; j++) {
		t += _dt;

		if(params.integratorType == IntegratorTypeEuler) {
			pair xP = xPrime(t,x);
			x.a += xP.a * _dt;
			x.b += xP.b * _dt;
		}
		
		else if(params.integratorType == IntegratorTypeRungeKutta)
			x = getNextRungeKuttaPairValue(x, t, xPrime, _dt);
		
		// Clamp the population size at zero
		if(x.a < 0.0) x.a = 0.0;
		if(x.b < 0.0) x.b = 0.0;
		// Clamp the population max at 1e100 just like the original PopBio did.
		if(x.a > 1e100) x.a = 1e100;
		if(x.b > 1e100) x.b = 1e100;
		
		x1Table[j] = x.a;
		x2Table[j] = x.b;
		
		printf("%lf\t\%lf\t\%lf\n", t, x.a, x.b);

		// Stabilization is screwing the Holling models up.
		// We'll add it back in later if we really need it.
		/*
		if( fabs(x1Table[j]-x1Table[j-1]) / _dt < params.epsilon &&
		    fabs(x2Table[j]-x2Table[j-1]) / _dt < params.epsilon ) {
			stabilized = 1;
			break;
		}
		*/
	}
	
	printf("END DATA\n");
	
	// Stabilization is screwing the Holling models up.
	// We'll add it back in later if we really need it.
	/*
	if(stabilized)
		printf("\ncomment: Stabilized after %lf time periods\n", t);
	*/
	
	free(x1Table);
	free(x2Table);
	
	return 0;
}