//
//  PopGrowthDocument.m
//  PopBio
//
//  Created by Aaron Golden on 05/09/07.
//  Copyright Reed College 2007 . All rights reserved.
//

#import "PopGrowthDocument.h"
#import "PlotWindowController.h"
#import "PopBioController.h"
#import "FineTunePanel.h"
#import "NSArrayAdditions.h"
#import "NSDictionaryCurve.h"

#define kPopGrowthWindowMinWidth 295.0
#define kPopGrowthWindowMaxWidth 464.0
#define kPopGrowthWindowHeight 358.0

@implementation PopGrowthDocument

// ----------------
// init and dealloc
// ----------------

- (id)init
{
    self = [super init];
    if (self) {
		params.timeIntervals = 200;
		params.initialPopulation = 10;
		params.rateOfIncrease = 0.05;
		params.rateOfIncreaseErrorSize = 0.0;
		params.rateOfIncreaseErrorDistributionType = kDistributionFixed;
		params.growthTimeLag = 0.0;
		params.carryingCapacity = 100;
		params.carryingCapacityErrorSize = 0.0;
		params.carryingCapacityErrorDistributionType = kDistributionFixed;
		params.numberOfTrials = 1;
		params.modelType = kModelTypeExponential;
		params.differenceEquation = 0;
		params.timeToK = 0;
		params.multipleTrials = 0;
		params.integratorType = IntegratorTypeEuler;
		params.pointStep = 1.0;
    }
    return self;
}

// --------------------------------------------------
// implementation of the document-specific parameters
// --------------------------------------------------

- (void)synchronizeParamsToUserInterface
{
	params.timeIntervals = [timeIntervalsTextField doubleValue];
	params.initialPopulation = [initialPopulationTextField doubleValue];
	params.rateOfIncrease = [rateOfIncreaseTextField doubleValue];
	params.rateOfIncreaseErrorSize = [rateOfIncreaseErrorTextField doubleValue];
	params.rateOfIncreaseErrorDistributionType = [rateOfIncreaseErrorDistributionPopUpButton indexOfSelectedItem];
	params.differenceEquation = [differenceEquationCheckBox state];
	params.growthTimeLag = [growthTimeLagTextField doubleValue];
	params.carryingCapacity = [carryingCapacityTextField doubleValue];
	params.carryingCapacityErrorSize = [carryingCapacityErrorTextField doubleValue];
	params.carryingCapacityErrorDistributionType = [carryingCapacityErrorDistributionPopUpButton indexOfSelectedItem];
	params.numberOfTrials = [numberOfTrialsTextField intValue];
	params.modelType = [modelSelectionMatrix selectedRow];
	params.differenceEquation = [differenceEquationCheckBox state];
	params.timeToK = [timeToKCheckBox state];
	params.multipleTrials = params.numberOfTrials > 1;
	params.integratorType = [fineTunePanel integratorType];
	params.pointStep = [fineTunePanel pointStep];
}

- (void)synchronizeUserInterfaceToParams
{
	[timeIntervalsTextField setDoubleValue:params.timeIntervals];
	[initialPopulationTextField setDoubleValue:params.initialPopulation];
	[rateOfIncreaseTextField setDoubleValue:params.rateOfIncrease];
	[rateOfIncreaseErrorTextField setDoubleValue:params.rateOfIncreaseErrorSize];
	[rateOfIncreaseErrorDistributionPopUpButton selectItemAtIndex:params.rateOfIncreaseErrorDistributionType];
	[rateOfIncreaseErrorTextField setEnabled:params.rateOfIncreaseErrorDistributionType != kDistributionFixed];
	[growthTimeLagTextField setDoubleValue:params.growthTimeLag];
	[carryingCapacityTextField setDoubleValue:params.carryingCapacity];
	[carryingCapacityErrorTextField setDoubleValue:params.carryingCapacityErrorSize];
	[carryingCapacityErrorDistributionPopUpButton selectItemAtIndex:params.carryingCapacityErrorDistributionType];
	[carryingCapacityErrorTextField setEnabled:params.carryingCapacityErrorDistributionType != kDistributionFixed];
	[numberOfTrialsTextField setIntValue:params.numberOfTrials];
	[exponentialButton setState:params.modelType == kModelTypeExponential];
	[logisiticButton setState:params.modelType == kModelTypeLogistic];
	[growthTimeLagTextField setEnabled:(params.modelType == kModelTypeLogistic)];
	[growthTimeLagLabel setTextColor:(params.modelType == kModelTypeLogistic)? [NSColor controlTextColor] : [NSColor disabledControlTextColor]];
	[carryingCapacityTextField setEnabled:(params.modelType == kModelTypeLogistic)];
	[carryingCapacityLabel setTextColor:(params.modelType == kModelTypeLogistic)? [NSColor controlTextColor] : [NSColor disabledControlTextColor]];
	[differenceEquationCheckBox setState:params.differenceEquation];
	[timeToKCheckBox setState:params.timeToK];
	[timeToKCheckBox setEnabled:(params.modelType == kModelTypeLogistic)];
	[fineTunePanel setIntegratorType:params.integratorType];
	[fineTunePanel setPointStep:params.pointStep];
		
	[self doGrowWindow];
}

- (NSArray*)argumentsForEvaluator
{
	return @[[NSString stringWithFormat:@"%d", kExperimentTypePopGrowth],
		[NSString stringWithFormat:@"%lf", params.timeIntervals],
		[NSString stringWithFormat:@"%lf", params.initialPopulation],
		[NSString stringWithFormat:@"%lf", params.rateOfIncrease],
		[NSString stringWithFormat:@"%lf", [stochasticButton state]? params.rateOfIncreaseErrorSize : 0.0],
		[NSString stringWithFormat:@"%d", params.rateOfIncreaseErrorDistributionType],
		[NSString stringWithFormat:@"%d", params.differenceEquation],
		[NSString stringWithFormat:@"%lf", params.growthTimeLag],
		[NSString stringWithFormat:@"%lf", params.carryingCapacity],
		[NSString stringWithFormat:@"%lf", [stochasticButton state]? params.carryingCapacityErrorSize : 0.0],
		[NSString stringWithFormat:@"%d", params.carryingCapacityErrorDistributionType],
		[NSString stringWithFormat:@"%d", params.numberOfTrials],
		[NSString stringWithFormat:@"%d", params.timeToK],
		[NSString stringWithFormat:@"%d", params.numberOfTrials > 0],
		[NSString stringWithFormat:@"%d", params.modelType],
		[NSString stringWithFormat:@"%.8f", [[[NSUserDefaults standardUserDefaults] objectForKey:@"popGrowthCapacityOrExtinctionThreshold"] doubleValue]],
		[NSString stringWithFormat:@"%d", params.integratorType],
		[NSString stringWithFormat:@"%lf", params.pointStep],
		[NSString stringWithFormat:@"%d", params.randomSeed++]];
}

- (NSString *)parametersReport;
{
	NSMutableString *str = [NSMutableString string];
	[str appendFormat:@"Population Growth:\n\n"];
	[str appendFormat:@"Time Intervals: %ld\n", lrint(params.timeIntervals)];
	[str appendFormat:@"Initial Population Size: %ld\n", lrint(params.initialPopulation)];
	[str appendFormat:@"Rate of Increase: %.2f\n",params.rateOfIncrease];
	[str appendFormat:@"Rate of Increase Error Size: %.2f\n", params.rateOfIncreaseErrorSize];
	if(params.rateOfIncreaseErrorDistributionType == kDistributionFixed)
		[str appendFormat:@"\"Fixed\" distribution for rate of increase error (no error)\n"];
	else if(params.rateOfIncreaseErrorDistributionType == kDistributionUniform)
		[str appendFormat:@"Uniform error distribution for rate of increase\n"];
	else if(params.rateOfIncreaseErrorDistributionType == kDistributionNormal)
		[str appendFormat:@"Normal error distribution for reate of increase\n"];
	
	[str appendFormat:@"Growth Time Lag: %lf\n",params.growthTimeLag];
	[str appendFormat:@"Carrying Capacity: %ld\n", lrint(params.carryingCapacity)];
	[str appendFormat:@"Carrying Capacity Error Size: %.2f\n", params.carryingCapacity];
	if(params.carryingCapacityErrorDistributionType == kDistributionFixed)
		[str appendFormat:@"\"Fixed\" distribution for carrying capacity error (no error)\n"];
	else if(params.carryingCapacityErrorDistributionType == kDistributionUniform)
		[str appendFormat:@"Uniform error distribution for carrying capacity\n"];
	else if(params.carryingCapacityErrorDistributionType == kDistributionNormal)
		[str appendFormat:@"Normal error distribution for carrying capacity\n"];
		
	[str appendFormat:@"Number of Trials: %d\n",params.numberOfTrials];
	[str appendFormat:@"Time to K: %@\n",params.timeToK? @"Yes" : @"No"];
	[str appendFormat:@"Number of Trials: %d\n",params.numberOfTrials];
	[str appendFormat:@"Model Type: %@\n", params.modelType==kModelTypeExponential? @"Exponential" : @"Logisitc"];
		
	[str appendFormat:@"Epsilon: %.8f\n",[[[NSUserDefaults standardUserDefaults] objectForKey:@"popGrowthCapacityOrExtinctionThreshold"] doubleValue]];	
	
	if(params.integratorType == IntegratorTypeRungeKutta)
		[str appendFormat:@"Runge-Kutta Integration\n"];
	else if(params.integratorType == IntegratorTypeEuler)
		[str appendFormat:@"Euler forward Integration\n"];
	if(!params.differenceEquation){
		[str appendFormat:@"Difference equations avoided when possible"];
	}

	return str;
}

// -----------------------------------------------------
// document-specific implementation of finishCalculation
// -----------------------------------------------------

- (void)finishCalculation
{
	NSArray *timeIntervalsData = nil;
	NSArray *populationData = nil;

	NSMutableArray *trialStrings = [[NSMutableArray alloc] initWithArray:[evaluatorOutput componentsSeparatedByString:@"TRIAL_DONE\n"]];
	[trialStrings removeLastObject]; // Kill the "DONE" entry.
	int trialNumber;
	for(trialNumber = 0; trialNumber < [trialStrings count]; trialNumber++){
		// Start out by getting an array of the new line separated
		// rows of data.  Remove the last row -- if the simulation
		// completed then the last row is DONE.  If the simulation
		// was killed early, then we don't know whether or not the
		// last row is complete, so just don't use it.
		NSMutableArray *rows = [[NSMutableArray alloc] initWithArray:[trialStrings[trialNumber] componentsSeparatedByString:@"\n"]];
		if([[rows lastObject] isEqualToString:@""])
			[rows removeLastObject];
		
		//[rows removeObjectAtIndex:[rows count]-1];
		
		// Every data set is an NSDictionary with an NSString tied to
		// @"title" and an NSArray of NSNumbers tied to @"data".
		
		// Clear data sets and generate a new data set
		// NSDictionary for each column in the first
		// output row, giving each new data set a title
		// and an empty array of data.
		NSMutableArray *dataSets = [[NSMutableArray alloc] init];
		NSArray *titles = [rows[0] componentsSeparatedByString:@"\t"];
		int j;
		for(j = 0; j < [titles count]; j++) {
			NSDictionary *newDataSet = @{@"title": titles[j], @"data": [[NSMutableArray alloc] init]};
			[dataSets addObject:newDataSet];
		}

		for(j = 1; j < [rows count]; j++){
			NSArray *numberStrings = [rows[j] componentsSeparatedByString:@"\t"];
			int k;
			for(k = 0; k < [numberStrings count]; k++){
				[dataSets[k][@"data"] addObject:@([numberStrings[k] doubleValue])];
			}
		}
		
		[trials addObject:@{@"params": [self parametersDictionary], @"dataSets": dataSets}];
	}
	
	NSArray *dataSets = [trials lastObject][@"dataSets"];

	// Get the data array for each of the data sets: time-intervals, population size,
	// that aught to exist because this was a PopGrowth run.
	timeIntervalsData = nil;
	populationData = nil;
	int j;
	for(j = 0; j < [dataSets count]; j++){
		id title = dataSets[j][@"title"];
		if(title && [title isKindOfClass:[NSString class]]){
			if([title isEqualToString:@"TimeIntervals"])
				timeIntervalsData = dataSets[j][@"data"];
			else if([title isEqualToString:@"Population"])
				populationData = dataSets[j][@"data"];
		}
	}
	
	// If any of Generations, Frequency, or Fitness were missing from the data
	// we got out of the backend, then something is very wrong and we need
	// to alert the user.
	if(!timeIntervalsData || !populationData){
		[[NSAlert alertWithMessageText:@"Data are missing."
			defaultButton:@"OK" alternateButton:nil otherButton:nil
			informativeTextWithFormat:@"An error has occurred.  The evaluator did not return all of the data required by the experiment."]
			beginSheetModalForWindow:[self windowForSheet] modalDelegate:nil didEndSelector:nil contextInfo:nil];
		return;
	}
	
	NSMutableString *commentString = [NSMutableString string];
	int timeToK = -1;
	if(params.timeToK && params.modelType == kModelTypeLogistic){
		timeToK = [self timeToKWithTimeIntervals:timeIntervalsData andPopulationData:populationData];
		if(timeToK != -1)
			[commentString setString:[NSString stringWithFormat:@"Reached K after %d time intervals", timeToK]];
	}
	if(timeToK == -1){
		double epsilon = [[[NSUserDefaults standardUserDefaults] objectForKey:@"popGrowthCapacityOrExtinctionThreshold"] doubleValue];
		double finalPopulation = [[populationData lastObject] doubleValue];
		if(finalPopulation < epsilon){
			int j;
			for(j = [populationData count]-2; j >= 0; j--){
				if([populationData[j] doubleValue] >= epsilon){
					[commentString setString:[NSString stringWithFormat:@"Extinction after %lf time intervals", [timeIntervalsData[j+1] doubleValue]]];
					break;
				}
			}
		}
		else{
			[commentString setString:[NSString stringWithFormat:@"Final population size = %g", finalPopulation]];
			
			if(params.modelType == kModelTypeExponential && (params.rateOfIncreaseErrorSize == 0.0 || params.rateOfIncreaseErrorDistributionType == kDistributionFixed)){
				[commentString appendFormat:@", doubling time = %lf", log(2.0)/params.rateOfIncrease];
			}
		}
	}
		
	// Create a persistent textual representation of the data
	// This is probably better created on the fly, but this allows passing
	// it to objects (PlotView, here) that don't have a reference to this object
	_parametersReport = [self parametersReport];
	_resultsReport = [self resultsReportWithTimeIntervalsData:timeIntervalsData populationData:populationData];
	
	// set persistent dictionaries with the information gathered.
	//if ( _geneticsResults){ [_geneticsResults release];}
	
	int timeIntervalsMax = (int)ceil([[timeIntervalsData lastObject] doubleValue]);
	int timeIntervalsTicks;
	if(10 <= timeIntervalsMax){
		timeIntervalsTicks = 10;
		timeIntervalsMax += timeIntervalsTicks - timeIntervalsMax % timeIntervalsTicks;
	}
	else
	 timeIntervalsTicks = timeIntervalsMax;
	
	//int popMax = (int)round([[populationData maximumNumber] doubleValue]);
	//int popTicks = 5;
	//popMax += popTicks - popMax % popTicks;

	double popMax = [[populationData maximumNumber] doubleValue];

	_populationResults = @{@"long-title": @"Population v. Time",
		@"short-title": @"Population v. Time",
		@"x-axis-label": @"Time Intervals",
		@"y-axis-label": @"Population",
		@"x-min": @0,
		@"x-max": @(timeIntervalsMax),
		@"y-min": @0,
		@"y-max": @(popMax),
		@"x-ticks": @(timeIntervalsTicks),
		@"y-ticks": @10,
		@"curves": @[[NSDictionary curveWithX:timeIntervalsData Y:populationData]],
		@"comment": commentString,
		@"parameters-report": _parametersReport, 
		@"results-report": _resultsReport};
	
	// Now spawn a plot window with the data from this run.
	
	// add the plotWindowController to ourselves in case it was removed by the user
	// closing the window manually.
	[self addWindowController:plotWindowController];
	// Set the title of the new plot window (not the title of the plot, just the title of the window holding the plot).
	[[plotWindowController window] setTitle:[self titleForPlotWindow]];
	
	// Note that the plotWindowController expects that certain key/value pairs will be available in any "plot"
	// dictionary.  The keys are: "long-title" (to go above plot), "short-title" to go in pop-up button, "x-axis-label",
	// "y-axis-label", "x-data", "y-data", "x-min", "x-max", "y-min", "y-max", and "comment".
	//[plotWindowController setPlots:[NSArray arrayWithObjects:_geneticsResults, _meanFitnessResults, _fitnessResults, nil]];

	[plotWindowController setPlots:@[_populationResults]];

	// Send the plotWindowController the selectPlot message.
	// This merely causes plotWindowController to check the status
	// of the plot selection pop-up-menu and display whichever plot
	// is selected.
	[plotWindowController selectPlot:nil];
	
	// Finally, after everything is initialized, show the window.
	hasPlotData = YES;
	[plotWindowController showWindow:self];
	
	[multipleTrialsResultsTable reloadData];
	[multipleTrialsWindow setTitle:[self titleForMultipleTrialsWindow]];
	// Show the results window if we're doing multiple trials
	if(params.multipleTrials && params.numberOfTrials > 1){
		[multipleTrialsWindow makeKeyAndOrderFront:nil];
		[[NSNotificationCenter defaultCenter] addObserver:self
			selector:@selector(windowWillClose:)
			name:NSWindowWillCloseNotification
			object:multipleTrialsWindow];
	}
	// If this wasn't a multiple trials run, then cull the last trial that we
	// added off the list (above), because the user probably doesn't want this
	// trial in the list.
	else{
		[trials removeLastObject];
	}

    [multipleTrialsResultsTable reloadData];
}

// -----------------------
// miscellaneous functions
// -----------------------

- (IBAction)helpButton:(id)sender
{
	[PBController openHelp:self];
}

- (NSString*)titleForPlotWindow
{
	return [NSString stringWithFormat:@"Growth Graphics: %@", [self displayName]];
}

- (NSString*)titleForMultipleTrialsWindow
{
	return [NSString stringWithFormat:@"%@ Data", [self displayName]];
}

- (NSString *)windowNibName
{
    return @"PopGrowthDocument";
}

- (NSData *)dataOfType:(NSString *)typeName error:(NSError **)outError
{
	[self synchronizeParamsToUserInterface];
	return [NSData dataWithBytes:&params length:sizeof(PopGrowthParameters)];
}

- (BOOL)readFromData:(NSData *)data ofType:(NSString *)typeName error:(NSError **)outError
{
	memcpy(&params, [data bytes], sizeof(PopGrowthParameters));
	[self synchronizeUserInterfaceToParams];
    return YES;
}

- (void)setFileName:(NSString*)fileName
{
	[super setFileName:fileName];
	[multipleTrialsWindow setTitle:[self titleForMultipleTrialsWindow]];
	[multipleTrialsWindow display];
}

- (void)doGrowWindow
{
	NSWindow *theWindow = [self windowForSheet];
	NSRect frame = [theWindow frame];

	// We're #defing the min and max sizes here instead of using
	// the IB settings because a bug in IB is making it impossible
	// for the min and max sizes to have the same height, though
	// they should have it.
	NSSize newSize;
	if([stochasticDeterministicMatrix selectedRow] == 1)
		newSize = NSMakeSize(kPopGrowthWindowMaxWidth,kPopGrowthWindowHeight);
	else
		newSize = NSMakeSize(kPopGrowthWindowMinWidth,kPopGrowthWindowHeight);

	//[theWindow setFrame:NSMakeRect(frame.origin.x, frame.origin.y, newSize.width, newSize.height) display:YES animate:YES];
	frame.size = newSize;
	[theWindow setFrame:frame display:YES animate:YES];
}

- (IBAction)showMultipleTrialsWindow:(id)sender
{
	[multipleTrialsWindow makeKeyAndOrderFront:sender];
}

- (IBAction)saveMultipleTrialsTableAsText:(id)sender
{
	NSMutableString *outString = [[NSMutableString alloc] initWithString:@"N(0)\tr\tK\tTime Lag\tt\tN(t)\n"];
	int j;
	for(j = 0; j < [trials count]; j++){
		NSDictionary *theParams = trials[j][@"params"];
		double rateOfIncrease = [theParams[@"rateOfIncrease"] doubleValue];
		double rateOfIncreaseErrorSize = [theParams[@"rateOfIncreaseErrorSize"] doubleValue];
		int rateOfIncreaseErrorDistributionType = [theParams[@"rateOfIncreaseErrorDistributionType"] doubleValue];
		double carryingCapacity = [theParams[@"carryingCapacity"] doubleValue];
		double carryingCapacityErrorSize = [theParams[@"carryingCapacityErrorSize"] doubleValue];
		int carryingCapacityErrorDistributionType = [theParams[@"carryingCapacityErrorDistributionType"] doubleValue];

		NSString *rateOfIncreaseString;
		NSString *carryingCapacityString;
		unsigned short int plusMinus = 0x00B1;
		if(rateOfIncreaseErrorSize > 0.0 && rateOfIncreaseErrorDistributionType != kDistributionFixed)
			rateOfIncreaseString = [NSString stringWithFormat:@"%.3lf%@%.3lf %@", rateOfIncrease, [NSString stringWithCharacters:&plusMinus length:1], rateOfIncreaseErrorSize,
						rateOfIncreaseErrorDistributionType==kDistributionUniform? @"(Uniformly distributed)" : @"(Normally distributed)"];
		else
			rateOfIncreaseString = [NSString stringWithFormat:@"%lf", rateOfIncrease];
			
		if(carryingCapacityErrorSize > 0.0 && carryingCapacityErrorDistributionType != kDistributionFixed)
			carryingCapacityString = [NSString stringWithFormat:@"%.3lf%@%.3lf %@", carryingCapacity, [NSString stringWithCharacters:&plusMinus length:1], carryingCapacityErrorSize,
						carryingCapacityErrorDistributionType==kDistributionUniform? @"(Uniformly distributed)" : @"(Normally distributed)"];
		else
			carryingCapacityString = [NSString stringWithFormat:@"%lf", carryingCapacity];
		
		[outString appendString:[NSString stringWithFormat:@"%lf\t%@\t%@\t%lf\t%lf\t%lf\n",
			[theParams[@"initialPopulation"] doubleValue],
			rateOfIncreaseString,
			carryingCapacityString,
			[theParams[@"growthTimeLag"] doubleValue],
			[self finalTimeInTrialIndex:j],
			[self finalPopulationInTrialIndex:j]]];
	}

	NSString *defaultFilename = [NSString stringWithFormat:@"%@ Trials", [self displayName]];
	NSSavePanel *sp = [NSSavePanel savePanel];
    [sp setAllowedFileTypes:@[@"txt"]];
    [sp setDirectoryURL:nil];
    [sp setNameFieldStringValue:defaultFilename];
    int runResult = [sp runModal];
	if (runResult == NSOKButton) {
        NSError *writeError = nil;
		if (![outString writeToURL:[sp URL] atomically:YES encoding:NSUTF8StringEncoding error:&writeError]){
			[[NSAlert alertWithMessageText:@"An error occurred while saving."
				defaultButton:@"OK" alternateButton:nil otherButton:nil
				informativeTextWithFormat:@"An error occurred while attempting to save the table: %@", [writeError localizedDescription]]
				beginSheetModalForWindow:multipleTrialsWindow modalDelegate:nil didEndSelector:nil contextInfo:nil];
		}
	}
	
	[saveMenuPopUpButton selectItemAtIndex:0];
}

- (NSDictionary*)parametersDictionary
{
	return @{@"timeIntervals": @(params.timeIntervals),
		@"initialPopulation": @(params.initialPopulation),
		@"rateOfIncrease": @(params.rateOfIncrease),
		@"rateOfIncreaseErrorSize": @(params.rateOfIncreaseErrorSize),
		@"rateOfIncreaseErrorDistributionType": @(params.rateOfIncreaseErrorDistributionType),
		@"growthTimeLag": @(params.growthTimeLag),
		@"carryingCapacity": @(params.carryingCapacity),
		@"carryingCapacityErrorSize": @(params.carryingCapacityErrorSize),
		@"carryingCapacityErrorDistributionType": @(params.carryingCapacityErrorDistributionType),
		@"numberOfTrials": @(params.numberOfTrials),
		@"differenceEquation": @(params.differenceEquation),
		@"timeToK": @(params.timeToK),
		@"modelType": @(params.modelType)};
}

- (int)timeToKWithTimeIntervals:(NSArray*)timeIntervalsData andPopulationData:(NSArray*)populationData
{
	double epsilon = [[[NSUserDefaults standardUserDefaults] objectForKey:@"popGrowthCapacityOrExtinctionThreshold"] doubleValue];
	int j;
	for(j = 0; j < [timeIntervalsData count] && j < [populationData count]; j++){
		double p = [populationData[j] doubleValue];
		if(ABS(p - params.carryingCapacity) < epsilon ||
			p > params.carryingCapacity)
			return [timeIntervalsData[j] doubleValue];
	}
	return -1;
}

- (NSString*)resultsReportWithTimeIntervalsData:(NSArray*)timeIntervals populationData:(NSArray*)populationData
{
	NSMutableString *timeIntervalsVPopulationString = [NSMutableString string];
	
	int i;
	for (i=0; i<[timeIntervals count]; i++) {
		[timeIntervalsVPopulationString appendFormat:@"%.3f %.3f\n", 
			[timeIntervals[i] doubleValue], [populationData[i] doubleValue] ];
	}
	
	return [[self parametersReport]
		stringByAppendingFormat:@"\nTime Intervals v. Population\n%@", timeIntervalsVPopulationString];
}

- (NSInteger)numberOfRowsInTableView:(NSTableView *)tableView
{
	return [trials count];
}

- (double)finalPopulationInTrialIndex:(int)index
{
	NSArray *dataSets = trials[index][@"dataSets"];
	NSArray *populationData = nil;
	int j;
	for(j = 0; j < [dataSets count]; j++){
		id title = dataSets[j][@"title"];
		if(title && [title isKindOfClass:[NSString class]] && [title isEqualToString:@"Population"])
			populationData = dataSets[j][@"data"];
	}

	// We're leaving these as NSLog's instead of user level alerts,
	// because in the event that we get a lot of them, alerts would
	// make the app unusable when all we really want is a log that
	// something out of the ordinary happened.
	if(!populationData){
		NSLog(@"No populationData data found for trial %d", index);
		return 0;
	}
	
	return [[populationData lastObject] doubleValue];
}

- (double)finalTimeInTrialIndex:(int)index
{
	NSArray *dataSets = trials[index][@"dataSets"];
	NSArray *timeData = nil;
	int j;
	for(j = 0; j < [dataSets count]; j++){
		id title = dataSets[j][@"title"];
		if(title && [title isKindOfClass:[NSString class]] && [title isEqualToString:@"TimeIntervals"])
			timeData = dataSets[j][@"data"];
	}

	// We're leaving these as NSLog's instead of user level alerts,
	// because in the event that we get a lot of them, alerts would
	// make the app unusable when all we really want is a log that
	// something out of the ordinary happened.
	if(!timeData){
		NSLog(@"No time data found for trial %d", index);
		return 0;
	}
	
	return [[timeData lastObject] doubleValue];
}

- (int)doublingTimeInTrialIndex:(int)index
{
	NSArray *dataSets = trials[index][@"dataSets"];
	NSArray *populationData = nil;
	int j;
	for(j = 0; j < [dataSets count]; j++){
		id title = dataSets[j][@"title"];
		if(title && [title isKindOfClass:[NSString class]] && [title isEqualToString:@"Population"])
			populationData = dataSets[j][@"data"];
	}

	// We're leaving these as NSLog's instead of user level alerts,
	// because in the event that we get a lot of them, alerts would
	// make the app unusable when all we really want is a log that
	// something out of the ordinary happened.
	if(!populationData){
		NSLog(@"No populationData data found for trial %d", index);
		return 0;
	}
	
	int *doublingTime = (int*)malloc(sizeof(int));
	int doublingTimesCount = 1;
	int time0 = 0;
	while(time0 < [populationData count]-1){
		int time1 = time0+1;
		double pop0 = [populationData[time0] doubleValue];
		double pop1 = [populationData[time1] doubleValue];
		while(pop1 < 2.0*pop0 && time1 < [populationData count]){
			time1++;
			if(time1 < [populationData count])
				pop1 = [populationData[time1] doubleValue];
		}
		if(time1 != [populationData count]){
			doublingTime[doublingTimesCount-1] = time1-time0;
			doublingTimesCount++;
			doublingTime = (int*)realloc(doublingTime, sizeof(int)*doublingTimesCount);		
		}
		time0 = time1;
	}
	if(doublingTimesCount > 1){
		int doublingTimeSum = 0;
		for(j = 0; j < doublingTimesCount-1; j++)
			doublingTimeSum += doublingTime[j];
		free(doublingTime);
		return (int)ceil((double)doublingTimeSum/(double)(doublingTimesCount-1));
	}
	else{
        free(doublingTime);
		return -1;
	}
}

- (id)tableView:(NSTableView *)tableView objectValueForTableColumn:(NSTableColumn *)tableColumn row:(NSInteger)row
{
	NSDictionary *theParams = trials[row][@"params"];
	double rateOfIncrease = [theParams[@"rateOfIncrease"] doubleValue];
	double rateOfIncreaseErrorSize = [theParams[@"rateOfIncreaseErrorSize"] doubleValue];
	int rateOfIncreaseErrorDistributionType = [theParams[@"rateOfIncreaseErrorDistributionType"] intValue];
	double carryingCapacity = [theParams[@"carryingCapacity"] doubleValue];
	double carryingCapacityErrorSize = [theParams[@"carryingCapacityErrorSize"] doubleValue];
	int carryingCapacityErrorDistributionType = [theParams[@"carryingCapacityErrorDistributionType"] intValue];
	unsigned short int plusMinus = 0x00B1;
	
	NSString *columnTitle = [[tableColumn headerCell] title];
	if([columnTitle isEqualToString:@"N(0)"])
		return theParams[@"initialPopulation"];
	else if([columnTitle isEqualToString:@"r"]){
		if(rateOfIncreaseErrorSize != 0.0)
			return [NSString stringWithFormat:@"%.3lf%@%.3lf %@", rateOfIncrease, [NSString stringWithCharacters:&plusMinus length:1], rateOfIncreaseErrorSize,
				rateOfIncreaseErrorDistributionType==kDistributionUniform? @"(Uniformly distributed)" : @"(Normally distributed)"];
		else
			return theParams[@"rateOfIncrease"];
	}
	else if([columnTitle isEqualToString:@"K"]){
		if(carryingCapacityErrorSize != 0.0)
			return [NSString stringWithFormat:@"%.3lf%@%.3lf %@", carryingCapacity, [NSString stringWithCharacters:&plusMinus length:1], carryingCapacityErrorSize,
				carryingCapacityErrorDistributionType==kDistributionUniform? @"(Uniformly distributed)" : @"(Normally distributed)"];
		else
			return theParams[@"carryingCapacity"];
	}
	else if([columnTitle isEqualToString:@"Time Lag"])
		return theParams[@"growthTimeLag"];
	else if([columnTitle isEqualToString:@"t"])
		return @([self finalTimeInTrialIndex:row]);
	//	return [theParams objectForKey:@"timeIntervals"];
	else if([columnTitle isEqualToString:@"N(t)"])
		return @([self finalPopulationInTrialIndex:row]);
	
	return @"Unknown column";
}

@end

@implementation PopGrowthDocument(NSWindowNotifications)

- (void)windowWillClose:(NSNotification *)notification
{
	// TODO: Maybe we should alert the user and ask
	// if they want to save the trials data?
	if([notification object] == multipleTrialsWindow){
		[trials removeAllObjects];
		[multipleTrialsResultsTable reloadData];
	}
}

@end
