/*
 *  PopGrowth.c
 *  PopBio
 *
 *  Created by Aaron Golden on 05/09/07.
 *  This work is provided under the terms of the Educational Community License 1.0, a copy of which is included with the source code.
 *
 */
 
#include <stdio.h>
#include <math.h>
#include <stdlib.h>

#include "Backend.h"
#include "PopGrowth.h"

// Algorithm from Leva -- ACM Transactions on Mathematical Software, Vol. 18, No. 4, December 1992, 449-453
double randomFromNormalDistribution(void)
{
	double u,v,x,y,Q;

start:
	u = (double)random() / (double)RAND_MAX;
	v = (double)random() / (double)RAND_MAX;
	v = 1.7156*(v-0.5);
	x = u-0.449871;
	y = ABS(v)+0.386595;
	Q = x*x+y*(0.19600*y-0.25472*x);
	if(Q < 0.27597)
		goto end;
	else if(Q > 0.27846 || (v*v > -4.0*u*u))
		goto start;
end:
	return v/u;
}

#define SHOULD_HALT (params.modelType == kModelTypeLogistic && params.timeToK && (ABS(populationSize-params.carryingCapacity) < params.epsilon || populationSize >= params.carryingCapacity)) || populationSize < params.epsilon

int runPopGrowth(int argc, char *argv[])
{
	if(argc < 19){
		printf("PopGrowth requires parameters:  <timeIntervals> <initial population size> \
		<rate of increase> <rate of increase error size> <rate of increase error distribution>\
		<difference equation> \
		<growth time lag> \
		<carrying capacity> <carrying capacity error size> <carrying capacity error distribution>\
		<number of trials> <difference equation> <time to K> <multiple trials> <model type> <epsilon>\
		<integrator type> <point step> <random seed>\n");
		return -1;
	}

	PopGrowthParameters params;
	sscanf(argv[2], "%lf", &params.timeIntervals);
	sscanf(argv[3], "%lf", &params.initialPopulation);
	sscanf(argv[4], "%lf", &params.rateOfIncrease);
	sscanf(argv[5], "%lf", &params.rateOfIncreaseErrorSize);
	sscanf(argv[6], "%d", &params.rateOfIncreaseErrorDistributionType);
	if(params.rateOfIncreaseErrorDistributionType == kDistributionFixed)
		params.rateOfIncreaseErrorSize = 0.0;
	
	sscanf(argv[7], "%d", &params.differenceEquation);
	sscanf(argv[8], "%lf", &params.growthTimeLag);	

	sscanf(argv[9], "%lf", &params.carryingCapacity);
	sscanf(argv[10], "%lf", &params.carryingCapacityErrorSize);
	sscanf(argv[11], "%d", &params.carryingCapacityErrorDistributionType);
	if(params.carryingCapacityErrorDistributionType == kDistributionFixed)
		params.carryingCapacityErrorSize = 0.0;
		
	sscanf(argv[12], "%d", &params.numberOfTrials);
	sscanf(argv[13], "%d", &params.timeToK);
	sscanf(argv[14], "%d", &params.multipleTrials);
	sscanf(argv[15], "%d", &params.modelType);
	sscanf(argv[16], "%lf", &params.epsilon);
	sscanf(argv[17], "%d", &params.integratorType);
	sscanf(argv[18], "%lf", &params.pointStep);
	sscanf(argv[19], "%d", &params.randomSeed);
	
	srandom(params.randomSeed);
	
	if(!params.multipleTrials)
		params.numberOfTrials = 1;
	// If the user doesn't demand a difference equation, stochasticity is off, and there is no growth time lag,
	// then we can avoid integrators entirely and just compute the population at
	// each time interval directly.
	if(!params.differenceEquation && (params.rateOfIncreaseErrorDistributionType == kDistributionFixed ||
		params.rateOfIncreaseErrorSize == 0.0) && (params.carryingCapacityErrorDistributionType == kDistributionFixed ||
		params.carryingCapacityErrorSize == 0.0) && params.growthTimeLag == 0.0){
		params.integratorType = IntegratorTypeNone;
	}

	double *populationList = (double*)malloc(sizeof(double)*((int)ceil(params.timeIntervals/params.pointStep)+1));
	int trialNumber;
	for(trialNumber = 1; trialNumber <= params.numberOfTrials; trialNumber++){
		printf("TimeIntervals\tPopulation\n");
		double populationSize = params.initialPopulation;
		double capacityPopulation = 0.0;
		double t;
		int j;
		for(t = 0.0, j = 0; t <= params.timeIntervals+params.pointStep*0.5 && j <= (int)ceil(params.timeIntervals/params.pointStep); t += params.pointStep, j++){
			double rateOfIncreaseError = 0;
			double carryingCapacityError = 0;
			if(params.rateOfIncreaseErrorDistributionType == kDistributionUniform)
				rateOfIncreaseError = (double)random()/(double)RAND_MAX*params.rateOfIncreaseErrorSize;
			else if(params.rateOfIncreaseErrorDistributionType == kDistributionNormal){
				rateOfIncreaseError = randomFromNormalDistribution()*params.rateOfIncreaseErrorSize;
			}
			else if(params.rateOfIncreaseErrorDistributionType == kDistributionFixed){
				rateOfIncreaseError = 0.0;
			}
			if(random() % 2)
				rateOfIncreaseError *= -1.0;
			
			if(params.carryingCapacityErrorDistributionType == kDistributionUniform)
				carryingCapacityError = (double)random()/(double)RAND_MAX*params.carryingCapacityErrorSize;
			else if(params.carryingCapacityErrorDistributionType == kDistributionNormal){
				carryingCapacityError = randomFromNormalDistribution()*params.carryingCapacityErrorSize;
			}
			else if(params.carryingCapacityErrorDistributionType == kDistributionFixed){
				carryingCapacityError = 0.0;
			}
			if(random() % 2)
				carryingCapacityError *= -1.0;
			
			populationList[j] = populationSize;
			printf("%lf\t\%lf\n", t, populationSize);
			// Kill the experiment here if we've reached the carrying capacity
			// or extinction.
			if(SHOULD_HALT){
				break;
			}
			
			if(t >= params.growthTimeLag){
				capacityPopulation = populationList[j-(int)round(params.growthTimeLag/params.pointStep)];
			}

			if(params.integratorType == IntegratorTypeEuler){
				if(params.modelType == kModelTypeExponential){
					populationSize += params.pointStep*(params.rateOfIncrease+rateOfIncreaseError)*populationSize;
				}
				else if(params.modelType == kModelTypeLogistic){				
					populationSize += params.pointStep*(params.rateOfIncrease+rateOfIncreaseError)*populationSize*(1.0-capacityPopulation/(params.carryingCapacity+carryingCapacityError));
				}
			}
			else if(params.integratorType == IntegratorTypeRungeKutta){
				double h = params.pointStep;
				double r = params.rateOfIncrease + rateOfIncreaseError;
				if(params.modelType == kModelTypeExponential){
					double k1 = r*populationSize;
					double k2 = r*(populationSize + 0.5*h*k1);
					double k3 = r*(populationSize + 0.5*h*k2);
					double k4 = r*(populationSize + h*k3);
					populationSize += h/6.0*(k1+2.0*k2+2.0*k3+k4);
				}
				else if(params.modelType == kModelTypeLogistic){
					double K = params.carryingCapacity + carryingCapacityError;
					
					double k1 = r*populationSize*(1.0-capacityPopulation/K);
					populationList[(int)round((t+0.5*h)/params.pointStep)] = populationSize+0.5*h*k1;
					if(t+0.5*h >= params.growthTimeLag)
						capacityPopulation = populationList[(int)round((t+0.5*h-params.growthTimeLag)/params.pointStep)];
						
					double k2 = r*(populationSize+0.5*h*k1)*(1.0-capacityPopulation/K);
					double k3 = r*(populationSize+0.5*h*k2)*(1.0-capacityPopulation/K);
					
					populationList[(int)round((t+h)/params.pointStep)] = populationSize+h*k3;
					if(t+h >= params.growthTimeLag)
						capacityPopulation = populationList[(int)round((t+h-params.growthTimeLag)/params.pointStep)];
					double k4 = r*(populationSize+h*k3)*(1.0-capacityPopulation/K);

					populationSize += h/6.0*(k1+2.0*k2+2.0*k3+k4);
				}
			}
			else if(params.integratorType == IntegratorTypeNone){
				if(params.modelType == kModelTypeExponential){
					populationSize = params.initialPopulation*exp(params.rateOfIncrease*(t+params.pointStep));
				}
				else if(params.modelType == kModelTypeLogistic){
					double A = params.initialPopulation;
					double e_to_rt = exp(params.rateOfIncrease*(t+params.pointStep));
					double K = params.carryingCapacity;					
					populationSize = A*e_to_rt*K/(A*(e_to_rt-1.0)+K);
				}
			}
			
			// Clamp the population size at zero
			if(populationSize < 0.0)
				populationSize = 0.0;
				
			// Clamp the population max at 1e100 just like the original PopBio did.
			if(populationSize > 1e100)
				populationSize = 1e100;
		}
		
		printf("TRIAL_DONE\n");
	}
	free(populationList);
	
	return 0;
}
