//
//  PopGenDocument.m
//  PopBio
//
//  Created by Aaron Golden on 3/23/07.
//  Copyright Reed College 2007 . All rights reserved.
//

#import "PopGenDocument.h"
#import "PlotWindowController.h"
#import "PopBioController.h"
#import "NSDictionaryCurve.h"

@implementation PopGenDocument

// ----------------
// init and dealloc
// ----------------

- (id)init
{
    self = [super init];
    if (self) {
		params.generations = 500;
		params.populationSize = 10;
		params.alleleAFrequency = 0.5;
		params.genotypeAAFitness = 1.0;
		params.genotypeAaFitness = 1.0;
		params.genotypeaaFitness = 1.0;
		//params.inbreedingCoefficient = 0.0;
		params.numberOfTrials = 1;
		params.multipleTrials = 0;
		params.drift = 0;
		params.randomSeed = time(0);
    }
    return self;
}

// --------------------------------------------------
// implementation of the document-specific parameters
// --------------------------------------------------

- (void)synchronizeParamsToUserInterface
{
	params.generations = [generationsTextField intValue];
	params.populationSize = [populationSizeTextField intValue];
	params.alleleAFrequency = [alleleAFrequencyTextField doubleValue];
	params.genotypeAAFitness = [genotypeAAFitnessTextField doubleValue];
	params.genotypeAaFitness = [genotypeAaFitnessTextField doubleValue];
	params.genotypeaaFitness = [genotypeaaFitnessTextField doubleValue];
	//params.inbreedingCoefficient = [inbreedingCoefficientTextField doubleValue];
	params.numberOfTrials = [numberOfTrialsTextField intValue];
	params.multipleTrials = (params.numberOfTrials > 1);
	params.simulateGeneticSelection = [selectionCheckBox state];
	params.drift = [driftCheckBox state];
}

- (void)synchronizeUserInterfaceToParams
{
	[generationsTextField setIntValue:params.generations];
	[populationSizeTextField setIntValue:params.populationSize];
	[alleleAFrequencyTextField setDoubleValue:params.alleleAFrequency];
	[genotypeAAFitnessTextField setDoubleValue:params.genotypeAAFitness];
	[genotypeAaFitnessTextField setDoubleValue:params.genotypeAaFitness];
	[genotypeaaFitnessTextField setDoubleValue:params.genotypeaaFitness];
	//[inbreedingCoefficientTextField setDoubleValue:params.inbreedingCoefficient];
	[numberOfTrialsTextField setIntValue:params.numberOfTrials];
	[selectionCheckBox setState:params.simulateGeneticSelection];
	[driftCheckBox setState:params.drift];
	
	// Call these toggles to make sure that everything
	// gets enabled/disabled to match the states of the
	// check boxes.
	[self selectionToggle:self];
	[self driftToggle:self];
}

- (NSArray*)argumentsForEvaluator
{
	int drift = params.drift;
	int selection = [selectionCheckBox state];
	BOOL useThreshold = [[[NSUserDefaults standardUserDefaults] objectForKey:@"popGeneticsStopOnFixationOrLoss"] boolValue];

	return @[[NSString stringWithFormat:@"%d", kExperimentTypePopGenetics],
		[NSString stringWithFormat:@"%d", params.generations],
		[NSString stringWithFormat:@"%d", params.populationSize],
		[NSString stringWithFormat:@"%lf", params.alleleAFrequency],
		[NSString stringWithFormat:@"%lf", selection? params.genotypeAAFitness : 1.0],
		[NSString stringWithFormat:@"%lf", selection? params.genotypeAaFitness : 1.0],
		[NSString stringWithFormat:@"%lf", selection? params.genotypeaaFitness : 1.0],
		//[NSString stringWithFormat:@"%lf", params.inbreedingCoefficient],
		[NSString stringWithFormat:@"%d", drift? params.numberOfTrials : 1],
		[NSString stringWithFormat:@"%d", drift? params.multipleTrials : 0],
		[NSString stringWithFormat:@"%d", params.drift],
		useThreshold? [NSString stringWithFormat:@"%.8f", [[[NSUserDefaults standardUserDefaults] objectForKey:@"popGeneticsFixationLossThreshold"] doubleValue]] : @"-1.0", 
		[NSString stringWithFormat:@"%d", params.randomSeed++]];
}

- (NSString *)parametersReport;
{
	NSMutableString *str = [NSMutableString string];
	[str appendFormat:@"Population Genetics:\n\n"];
	[str appendFormat:@"Generations: %d\n",params.generations];
	[str appendFormat:@"Population Size: %d\n",params.populationSize];
	[str appendFormat:@"Allele A Frequency: %.2f\n",params.alleleAFrequency];
	[str appendFormat:@"Genotype AA Fitness: %.2f\n",params.genotypeAAFitness];
	[str appendFormat:@"Genotype Aa Fitness: %.2f\n",params.genotypeAaFitness];
	[str appendFormat:@"Genotype aa Fitness: %.2f\n",params.genotypeaaFitness];
	//[str appendFormat:@"Inbreeding Coefficient: %.2f\n",params.inbreedingCoefficient];
	[str appendFormat:@"Number of Trials: %d\n",params.numberOfTrials];
	[str appendFormat:@"Multiple Trials?: %@\n",params.multipleTrials? @"Yes":@"No"];
	[str appendFormat:@"Simulate darwinian selection?: %@\n", params.simulateGeneticSelection? @"Yes":@"No"];
	[str appendFormat:@"Simulate stochastic drift?: %@\n", params.drift? @"Yes":@"No"];
	[str appendFormat:@"Epsilon: %.8f",[[[NSUserDefaults standardUserDefaults] objectForKey:@"popGeneticsFixationLossThreshold"] doubleValue]];
	return str;	
}

// -----------------------------------------------------
// document-specific implementation of finishCalculation
// -----------------------------------------------------

- (void)finishCalculation
{
	NSArray *generationsData = nil;
	NSArray *frequency1Data = nil;
	NSArray *frequency2Data = nil;
	NSArray *fitness1Data = nil;
	NSArray *fitness2Data = nil;

	NSMutableArray *trialStrings = [[NSMutableArray alloc] initWithArray:[evaluatorOutput componentsSeparatedByString:@"TRIAL_DONE\n"]];
	[trialStrings removeLastObject]; // Kill the "DONE" entry.
	int trialNumber;
	for(trialNumber = 0; trialNumber < [trialStrings count]; trialNumber++){
		// Start out by getting an array of the new line separated
		// rows of data.  Remove the last row -- if the simulation
		// completed then the last row is DONE.  If the simulation
		// was killed early, then we don't know whether or not the
		// last row is complete, so just don't use it.
		NSMutableArray *rows = [[NSMutableArray alloc] initWithArray:[trialStrings[trialNumber] componentsSeparatedByString:@"\n"]];
		//[rows removeObjectAtIndex:[rows count]-1];
		
		// Every data set is an NSDictionary with an NSString tied to
		// @"title" and an NSArray of NSNumbers tied to @"data".
		
		// Clear data sets and generate a new data set
		// NSDictionary for each column in the first
		// output row, giving each new data set a title
		// and an empty array of data.
		NSMutableArray *dataSets = [NSMutableArray array];
		NSArray *titles = [rows[1] componentsSeparatedByString:@"\t"];
		int j;
		for(j = 0; j < [titles count]; j++){
			[dataSets addObject:@{@"title": titles[j], @"data": [NSMutableArray array]}];
		}

		for(j = 2; j < [rows count] && ![rows[j] isEqualToString:@"Fitness-Set"]; j++){
			NSArray *numberStrings = [rows[j] componentsSeparatedByString:@"\t"];
			int k;
			for(k = 0; k < [numberStrings count]; k++){
				[dataSets[k][@"data"] addObject:@([numberStrings[k] doubleValue])];
			}
		}

		j++;
		if(j < [rows count]){
			titles = [rows[j] componentsSeparatedByString:@"\t"];
			int k;
			for(k = 0; k < [titles count]; k++){
				[dataSets addObject:@{@"title": titles[k], @"data": [NSMutableArray array]}];
			}
		}
		j++;
		for(; j < [rows count]; j++){
			NSArray *numberStrings = [rows[j] componentsSeparatedByString:@"\t"];
			int k;
			for(k = 0; k < [numberStrings count]; k++){
				// TODO: That literal +3 is really ugly, let's fix it some time.
				[dataSets[k+3][@"data"] addObject:@([numberStrings[k] doubleValue])];
			}
		}
		
		[trials addObject:@{@"params": [self parametersDictionary], @"dataSets": dataSets}];
	}
	
	NSArray *dataSets = [trials lastObject][@"dataSets"];

	// Get the data array for each of the data sets: generations, frequency, and fitness,
	// that aught to exist because this was a PopGenetics run.
	generationsData = nil;
	frequency1Data = nil;
	frequency2Data = nil;
	fitness1Data = nil;
	fitness2Data = nil;
	int j;
	for(j = 0; j < [dataSets count]; j++){
		id title = dataSets[j][@"title"];
		if(title && [title isKindOfClass:[NSString class]]){
			if([title isEqualToString:@"Generations"])
				generationsData = dataSets[j][@"data"];
			else if([title isEqualToString:@"Frequency1"])
				frequency1Data = dataSets[j][@"data"];
			else if([title isEqualToString:@"Frequency2"])
				frequency2Data = dataSets[j][@"data"];
			else if([title isEqualToString:@"Fitness1"])
				fitness1Data = dataSets[j][@"data"];
			else if([title isEqualToString:@"Fitness2"])
				fitness2Data = dataSets[j][@"data"];
		}
	}
	
	// If any of Generations, Frequency, or Fitness were missing from the data
	// we got out of the backend, then something is very wrong and we need
	// to alert the user.
	if(!generationsData || !frequency1Data || !frequency2Data || !fitness1Data || !fitness2Data){
		[[NSAlert alertWithMessageText:@"Data are missing."
			defaultButton:@"OK" alternateButton:nil otherButton:nil
			informativeTextWithFormat:@"An error has occurred.  The evaluator did not return all of the data required by the experiment."]
			beginSheetModalForWindow:[self windowForSheet] modalDelegate:nil didEndSelector:nil contextInfo:nil];
		return;
	}
	
	NSMutableString *commentString = [[NSMutableString alloc] init];
	BOOL fixation = NO;
	BOOL loss = NO;
	for(j = 0; j < [frequency1Data count]; j++){
		if([frequency1Data[j] doubleValue] < [[[NSUserDefaults standardUserDefaults] objectForKey:@"popGeneticsFixationLossThreshold"] doubleValue]){
			[commentString setString:[NSString stringWithFormat:@"Loss after %d generations", j]];
			fixation = YES;
			break;
		}
		else if(1.0-[frequency1Data[j] doubleValue] < [[[NSUserDefaults standardUserDefaults] objectForKey:@"popGeneticsFixationLossThreshold"] doubleValue]){
			[commentString setString:[NSString stringWithFormat:@"Fixation after %d generations", j]];
			loss = YES;
			break;
		}
	}
	if(!fixation && !loss)
		[commentString setString:@"No fixation or loss"];
		
	// Create a persistent textual representation of the data
	// This is probably better created on the fly, but this allows passing
	// it to objects (PlotView, here) that don't have a reference to this object
	_parametersReport = [self parametersReport];
	_resultsReport = [self resultsReportWithGenerationsData:generationsData 
		frequency1Data:frequency1Data fitness1Data:fitness1Data
		frequency2Data:frequency2Data fitness2Data:fitness2Data];
	
	// set persistent dictionaries with the information gathered.
	
	// Find a number of ticks that's relatively
	// small and divides generations count-1 evenly.
	// If no number between 4 and 50 divides
	// generations count, just use generations count-1.
	int generationsTicks;
	for(generationsTicks = 4; generationsTicks <= 50; generationsTicks++){
		if(([generationsData count]-1) % generationsTicks == 0){
			break;
		}
	}
	// If we didn't find anything just use the number of generations.
	if(generationsTicks == 51)
		generationsTicks = [generationsData count]-1;
	
	NSDictionary *_geneticsCurve =    [NSDictionary curveWithX:generationsData Y:frequency1Data];
	NSDictionary *_meanFitnessCurve = [NSDictionary curveWithX:generationsData Y:fitness1Data];
	NSDictionary *_fitnessCurve =     [NSDictionary curveWithX:frequency2Data  Y:fitness2Data];
	
	_geneticsResults = @{@"long-title": @"Allele Frequency v. Generations",
			@"short-title": @"Allele Frequency v. Time",
			@"x-axis-label": @"Generations",
			@"y-axis-label": @"Frequency of allele A",
			@"x-min": @0,
			@"x-max": @([generationsData count]-1UL),
			@"y-min": @0.0,
			@"y-max": @1.0,
			@"x-ticks": @(generationsTicks),
			@"y-ticks": @10,
			@"curves": @[_geneticsCurve],
			@"comment": commentString,
			@"parameters-report": _parametersReport,
			@"results-report": _resultsReport};
	
	_meanFitnessResults = @{@"long-title": @"Population Fitness v. Generations",
			@"short-title": @"Mean Fitness v. Time",
			@"x-axis-label": @"Generations",
			@"y-axis-label": @"Mean Population Fitness",
			@"x-min": @0,
			@"x-max": @([generationsData count]-1UL),
			@"y-min": @0.0,
			@"y-max": @1.0,
			@"x-ticks": @(generationsTicks),
			@"y-ticks": @10,
			@"curves": @[_meanFitnessCurve],
			@"comment": commentString,
			@"parameters": _parametersReport, 
			@"results-report": _resultsReport};
	
	_fitnessResults = @{@"long-title": @"Mean Population Fitness v. Allele Frequency",
			@"short-title": @"Mean Fitness v. Allele Frequency",
			@"x-axis-label": @"Allele Frequency",
			@"y-axis-label": @"Mean Population Fitness",
			@"x-min": @0.0,
			@"x-max": @1.0,
			@"y-min": @0.0,
			@"y-max": @1.0,
			@"x-ticks": @10,
			@"y-ticks": @10,
			@"curves": @[_fitnessCurve],
			@"comment": @"",
			@"parameters": _parametersReport, 
			@"results-report": _resultsReport};
	
	// Now spawn a plot window with the data from this run.
	// A PopGenetics plot window should have two plots available:
	// Genetics (selected by default) and Fitness.
	// The Genetics plot is Frequency of allele A v. Generations.
	// The Fitness plot is Mean Population Fitness v. Allele Frequency.
	
	// add the plotWindowController to ourselves in case it was removed by the user
	// closing the window manually.
	[self addWindowController:plotWindowController];
	// Set the title of the new plot window (not the title of the plot, just the title of the window holding the plot).
	[[plotWindowController window] setTitle:[self titleForPlotWindow]];
	
	// Note that the plotWindowController expects that certain key/value pairs will be available in any "plot"
	// dictionary.  The keys are: "long-title" (to go above plot), "short-title" to go in pop-up button, "x-axis-label",
	// "y-axis-label", "x-data", "y-data", "x-min", "x-max", "y-min", "y-max", and "comment".
	[plotWindowController setPlots:@[_geneticsResults, _meanFitnessResults, _fitnessResults]];


	// Send the plotWindowController the selectPlot message.
	// This merely causes plotWindowController to check the status
	// of the plot selection pop-up-menu and display whichever plot
	// is selected.
	[plotWindowController selectPlot:nil];
	
	// Finally, after everything is initialized, show the window.
	hasPlotData = YES;
	[plotWindowController showWindow:self];
	
	[multipleTrialsResultsTable reloadData];
	[multipleTrialsWindow setTitle:[self titleForMultipleTrialsWindow]];
	// Show the results window if we're doing multiple trials
	if(params.multipleTrials && params.numberOfTrials > 1){
		[multipleTrialsWindow makeKeyAndOrderFront:nil];
		[[NSNotificationCenter defaultCenter] addObserver:self
			selector:@selector(windowWillClose:)
			name:NSWindowWillCloseNotification
			object:multipleTrialsWindow];
	}
	// If this wasn't a multiple trials run, then cull the last trial that we
	// added off the list (above), because the user probably doesn't want this
	// trial in the list.
	else
		[trials removeLastObject];

    [multipleTrialsResultsTable reloadData];
}

// -----------------------
// miscellaneous functions
// -----------------------

- (IBAction)helpButton:(id)sender
{
	[PBController openHelp:self];
}

- (NSString*)titleForPlotWindow
{
	return [NSString stringWithFormat:@"Genetics Graphics: %@", [self displayName]];
}

- (NSString*)titleForMultipleTrialsWindow
{
	return [NSString stringWithFormat:@"%@ Data", [self displayName]];
}

- (NSString *)windowNibName
{
    return @"PopGenDocument";
}

- (NSData *)dataOfType:(NSString *)typeName error:(NSError **)outError
{
	[self synchronizeParamsToUserInterface];
	return [NSData dataWithBytes:&params length:sizeof(PopGenParameters)];
}

- (BOOL)readFromData:(NSData *)data ofType:(NSString *)typeName error:(NSError **)outError
{
	memcpy(&params, [data bytes], sizeof(PopGenParameters));
    return YES;
}

- (void)setFileName:(NSString*)fileName
{
	[super setFileName:fileName];
	[multipleTrialsWindow setTitle:[self titleForMultipleTrialsWindow]];
	[multipleTrialsWindow display];
}

- (IBAction)selectionToggle:(id)sender
{
	int enabled = [selectionCheckBox state];
	[genotypeAAFitnessLabel setTextColor:enabled? [NSColor controlTextColor] : [NSColor disabledControlTextColor]];
	[genotypeAAFitnessTextField setEnabled:enabled];
	[genotypeAaFitnessLabel setTextColor:enabled? [NSColor controlTextColor] : [NSColor disabledControlTextColor]];
	[genotypeAaFitnessTextField setEnabled:enabled];
	[genotypeaaFitnessLabel setTextColor:enabled? [NSColor controlTextColor] : [NSColor disabledControlTextColor]];
	[genotypeaaFitnessTextField setEnabled:enabled];    
}

- (IBAction)driftToggle:(id)sender
{
	int enabled = [driftCheckBox state];
	[populationSizeLabel setTextColor:enabled? [NSColor controlTextColor] : [NSColor disabledControlTextColor]];
	[populationSizeTextField setEnabled:enabled];
	[numberOfTrialsLabel setTextColor:enabled? [NSColor controlTextColor] : [NSColor disabledControlTextColor]];
	[numberOfTrialsTextField setEnabled:enabled];
}

- (IBAction)saveMultipleTrialsTableAsText:(id)sender
{
	NSMutableString *outString = [[NSMutableString alloc] initWithString:@"Allele A Freq\tPopulation Size\tAA Fitness\tAa Fitness\taa Fitness\tTime to Fix/Loss\tFixation/Loss\n"];
	int j;
	for(j = 0; j < [trials count]; j++){
		NSDictionary *theParams = trials[j][@"params"];
		[outString appendString:[NSString stringWithFormat:@"%.4f\t%d\t%.4f\t%.4f\t%.4f\t%d\t%@\n",
			[theParams[@"alleleAFrequency"] doubleValue],
			[theParams[@"populationSize"] intValue],
			[theParams[@"genotypeAAFitness"] doubleValue],
			[theParams[@"genotypeAaFitness"] doubleValue],
			[theParams[@"genotypeaaFitness"] doubleValue],
			[self timeToFixOrLossInTrialIndex:j],
			[self fixationInTrialIndex:j]? @"Fixation" : [self lossInTrialIndex:j]? @"Loss" : @"Undetermined"]];
	}

	NSString *defaultFilename = [NSString stringWithFormat:@"%@ Trials", [self displayName]];
	NSSavePanel *sp = [NSSavePanel savePanel];
    [sp setAllowedFileTypes:@[@"txt"]];
    [sp setDirectoryURL:nil];
    [sp setNameFieldStringValue:defaultFilename];
    int runResult = [sp runModal];
	if (runResult == NSOKButton) {
        NSError *writeError = nil;
		if (![outString writeToURL:[sp URL] atomically:YES encoding:NSUTF8StringEncoding error:&writeError]) {
			[[NSAlert alertWithMessageText:@"An error occurred while saving."
                             defaultButton:@"OK" alternateButton:nil otherButton:nil
                 informativeTextWithFormat:@"An error occurred while attempting to save the table: %@", [writeError localizedDescription]]
                  beginSheetModalForWindow:multipleTrialsWindow modalDelegate:nil didEndSelector:nil contextInfo:nil];
		}
	}
	
	[saveMenuPopUpButton selectItemAtIndex:0];
}

- (IBAction)saveMultipleTrialsTableForLaTeX:(id)sender
{
	NSString *defaultFilename = [NSString stringWithFormat:@"%@ Trials", [self displayName]];

	NSMutableString *outString = [[NSMutableString alloc] initWithString:
                                   @"\\begin{table}[htbp]\n\\centering\n\\begin{tabular}{ccccccc}\nAllele A Freq & Population Size & AA Fitness & Aa Fitness & aa Fitness & Time to Fix/Loss & Fixation/Loss \\\\ \n"];
	int j;
	for(j = 0; j < [trials count]; j++){
		[outString appendString:[NSString stringWithFormat:@"%.4f & %d & %.4f & %.4f & %.4f & %d & %@ \\\\ \n",
			params.alleleAFrequency,
			params.populationSize,
			params.genotypeAAFitness,
			params.genotypeAaFitness,
			params.genotypeaaFitness,
			[self timeToFixOrLossInTrialIndex:j],
			[self fixationInTrialIndex:j]? @"Fixation" : [self lossInTrialIndex:j]? @"Loss" : @"Undetermined"]];
	}
	[outString appendString:[NSString stringWithFormat:@"\\end{tabular}\n\\caption{%@}\n\\end{table}\n", defaultFilename]];
	
	NSSavePanel *sp = [NSSavePanel savePanel];
    [sp setAllowedFileTypes:@[@"tex"]];
    [sp setDirectoryURL:nil];
    [sp setNameFieldStringValue:defaultFilename];
    int runResult = [sp runModal];
	if (runResult == NSOKButton) {
        NSError *writeError = nil;
		if (![outString writeToURL:[sp URL] atomically:YES encoding:NSUTF8StringEncoding error:&writeError]){
			[[NSAlert alertWithMessageText:@"An error occurred while saving."
				defaultButton:@"OK" alternateButton:nil otherButton:nil
				informativeTextWithFormat:@"An error occurred while attempting to save the table: %@", [writeError localizedDescription]]
				beginSheetModalForWindow:multipleTrialsWindow modalDelegate:nil didEndSelector:nil contextInfo:nil];
		}
	}
	
	[saveMenuPopUpButton selectItemAtIndex:0];
}

- (NSDictionary*)parametersDictionary
{
	return @{@"generations": @(params.generations),
		@"populationSize": @(params.populationSize),
		@"alleleAFrequency": @(params.alleleAFrequency),
		@"genotypeAAFitness": @(params.genotypeAAFitness),
		@"genotypeAaFitness": @(params.genotypeAaFitness),
		@"genotypeaaFitness": @(params.genotypeaaFitness),
		//[NSNumber numberWithDouble:params.inbreedingCoefficient], @"inbreedingCoefficient",
		@"numberOfTrials": @(params.numberOfTrials),
		@"multipleTrials": @((BOOL)(params.numberOfTrials > 1)),
		@"simulateGeneticSelection": @(params.simulateGeneticSelection),
		@"drift": @(params.drift)};
}

- (NSDictionary *)fitnessResults;
{
	return _fitnessResults;
}

- (NSDictionary *)geneticsResults;
{	
	return _geneticsResults;
}

- (IBAction)showMultipleTrialsWindow:(id)sender
{
	[multipleTrialsWindow makeKeyAndOrderFront:sender];
}

- (NSString*)resultsReportWithGenerationsData:generationsData frequency1Data:frequency1Data fitness1Data:fitness1Data frequency2Data:frequency2Data fitness2Data:fitness2Data;
{
	int i;
	NSMutableString *geneticsVTimeString = [NSMutableString string];
	NSMutableString *fitnessVTimeString  = [NSMutableString string];
	NSMutableString *fitnessVGeneticsString  = [NSMutableString string];
	
	for (i=0; i<[generationsData count]; i++) {
		[geneticsVTimeString appendFormat:@"%.3f %.3f\n", 
			[generationsData[i] doubleValue], [frequency1Data[i] doubleValue] ];
	}
	
	for (i=0; i<MIN([generationsData count],[fitness1Data count]); i++) {
		[fitnessVTimeString appendFormat:@"%.3f %.3f\n", 
			[generationsData[i] doubleValue], [fitness1Data[i] doubleValue] ];
	}
	
	for (i=0; i<MIN([frequency2Data count],[fitness2Data count]); i++) {
		[fitnessVGeneticsString appendFormat:@"%.3f %.3f\n", 
			[frequency2Data[i] doubleValue], [fitness2Data[i] doubleValue] ];
	}
	
	return [[self parametersReport]
		stringByAppendingFormat:@"\n\nGenetics v. Time\n%@\nFitness v. Time\n%@\nFitness v. Genetics\n%@",
		geneticsVTimeString, fitnessVTimeString, fitnessVGeneticsString];
}

- (NSInteger)numberOfRowsInTableView:(NSTableView *)tableView
{
	return [trials count];
}

- (int)timeToFixOrLossInTrialIndex:(int)trialIndex
{
	NSArray *dataSets = trials[trialIndex][@"dataSets"];
	NSArray *generationsData = nil;
	int j;
	for(j = 0; j < [dataSets count]; j++){
		id title = dataSets[j][@"title"];
		if(title && [title isKindOfClass:[NSString class]] && [title isEqualToString:@"Generations"])
			generationsData = dataSets[j][@"data"];
	}

	// We're leaving these as NSLog's instead of user level alerts,
	// because in the event that we get a lot of them, alerts would
	// make the app unusable when all we really want is a log that
	// something out of the ordinary happened.
	if(!generationsData){
		NSLog(@"No generations data found for trial %d", trialIndex);
		return 0;
	}
	
	return [generationsData count]-1;
}

- (BOOL)fixationInTrialIndex:(int)trialIndex
{
	NSArray *dataSets = trials[trialIndex][@"dataSets"];
	NSArray *frequency1Data = nil;
	int j;
	for(j = 0; j < [dataSets count]; j++){
		id title = dataSets[j][@"title"];
		if(title && [title isKindOfClass:[NSString class]] && [title isEqualToString:@"Frequency1"])
			frequency1Data = dataSets[j][@"data"];
	}

	if(!frequency1Data){
		NSLog(@"No frequency1Data found for trial %d", trialIndex);
		return 0;
	}
	
	return 1.0 - [[frequency1Data lastObject] doubleValue] < [[[NSUserDefaults standardUserDefaults] objectForKey:@"popGeneticsFixationLossThreshold"] doubleValue];
}

- (BOOL)lossInTrialIndex:(int)trialIndex
{
	NSArray *dataSets = trials[trialIndex][@"dataSets"];
	NSArray *frequency1Data = nil;
	int j;
	for(j = 0; j < [dataSets count]; j++){
		id title = dataSets[j][@"title"];
		if(title && [title isKindOfClass:[NSString class]] && [title isEqualToString:@"Frequency1"])
			frequency1Data = dataSets[j][@"data"];
	}

	if(!frequency1Data){
		NSLog(@"No frequency1Data found for trial %d", trialIndex);
		return 0;
	}
	
	return [[frequency1Data lastObject] doubleValue] < [[[NSUserDefaults standardUserDefaults] objectForKey:@"popGeneticsFixationLossThreshold"] doubleValue];
}

- (id)tableView:(NSTableView *)tableView objectValueForTableColumn:(NSTableColumn *)tableColumn row:(NSInteger)row
{
	NSDictionary *theParams = trials[row][@"params"];

	NSString *columnTitle = [[tableColumn headerCell] title];
	if([columnTitle isEqualToString:@"Allele A Freq"])
		return theParams[@"alleleAFrequency"];
	else if([columnTitle isEqualToString:@"Population Size"])
		return theParams[@"populationSize"];
	else if([columnTitle isEqualToString:@"AA Fitness"])
		return theParams[@"genotypeAAFitness"];
	else if([columnTitle isEqualToString:@"Aa Fitness"])
		return theParams[@"genotypeAaFitness"];
	else if([columnTitle isEqualToString:@"aa Fitness"])
		return theParams[@"genotypeaaFitness"];
	else if([columnTitle isEqualToString:@"Time to Fix/Loss"])
		return @([self timeToFixOrLossInTrialIndex:row]);
	else if([columnTitle isEqualToString:@"Fixation/Loss"]){
		if([self fixationInTrialIndex:row])
			return @"Fixation";
		else if([self lossInTrialIndex:row])
			return @"Loss";
		else
			return @"Undetermined";
	}
	
	return @"Unknown column";
}

@end

@implementation PopGenDocument(NSWindowNotifications)

- (void)windowWillClose:(NSNotification *)notification
{
	// TODO: Maybe we should alert the user and ask
	// if they want to save the trials data?
	if([notification object] == multipleTrialsWindow){
		[trials removeAllObjects];
		[multipleTrialsResultsTable reloadData];
	}
}

@end
