//
//  PopBioDocument.m
//  PopBio
//
//  Created by Aaron Golden on 3/26/07.
//  This work is provided under the terms of the Educational Community License 1.0, a copy of which is included with the source code.
//

#import "PopBioDocument.h"
#import "PlotWindowController.h"

@implementation PopBioDocument

- (id)init
{
	self = [super init];
	if(self){		
		trials = [[NSMutableArray alloc] init];
		
		parametersWindowController = nil;
		plotWindowController = nil;
		hasPlotData = NO;
		calculationInProgress = NO;

		evaluatorOutput = nil;
		parametersReport = nil;
		generatedPlotCode = nil;
		fixedPlotCode = nil;
	}
	return self;
}

- (IBAction)beginCalculation:(id)sender
{
	if(calculationInProgress) {
		if(evaluator){
			if([evaluator isRunning])
				[evaluator terminate];
			[[NSNotificationCenter defaultCenter] postNotificationName:NSTaskDidTerminateNotification object:evaluator];
		}
		
		return;
	}
	
	// Save the parameters report for later use
	parametersReport = [self parametersReport];
		
	// Initialize an instance of the Evaluator program.
	// The instance will live until this isolated calculation
	// is completed.  It takes command line arguments for each
	// of the parameters.
	NSString *path = 0;
	path = [[NSBundle mainBundle] pathForAuxiliaryExecutable:@"Evaluator"];
	toPipe = [NSPipe pipe];
	fromPipe = [NSPipe pipe];
	toEvaluator = [toPipe fileHandleForWriting];
	fromEvaluator = [fromPipe fileHandleForReading];
	evaluator = [[NSTask alloc] init];
	[evaluator setLaunchPath:path];
	[evaluator setStandardOutput:fromPipe];
	[evaluator setStandardInput:toPipe];
	
	// Initialize a string to take the output of the evaluator
	evaluatorOutput = [[NSMutableString alloc] init];

	// Set up the command line arguments to feed the evaluator
	// the parameters of this system.	
	[self synchronizeParamsToUserInterface];
	[evaluator setArguments: [self argumentsForEvaluator]];
		
	[[NSNotificationCenter defaultCenter]
		addObserver:self selector:@selector(evaluatorDidTerminate:)
		name:NSTaskDidTerminateNotification object:evaluator];
	
	[self setCalculationInProgress:YES];
	[evaluator launch];
	
	[[NSNotificationCenter defaultCenter]
		addObserver:self selector:@selector(gotData:)
		name:NSFileHandleReadCompletionNotification object:fromEvaluator];
	[fromEvaluator readInBackgroundAndNotify];
}

- (void)evaluatorDidTerminate:(NSNotification*)notification
{
	[[NSNotificationCenter defaultCenter] removeObserver:self
		name:NSTaskDidTerminateNotification object:evaluator];
	[self setCalculationInProgress:NO];
}

- (void)gotData:(NSNotification*)notification
{
	NSData *data;
	NSString *str;
	
	data = [notification userInfo][NSFileHandleNotificationDataItem];
	str = [[NSString alloc] initWithData:data encoding:NSASCIIStringEncoding];

	// Check to see if the latest message from the evaluator was
	// its "DONE" message.
	int oldLength = [evaluatorOutput length];
	[evaluatorOutput appendString:str];
	int newLength = [evaluatorOutput length];
	NSRange startRange = [evaluatorOutput rangeOfString:@"EXPERIMENT_STARTING\n"];
	NSString *experimentEndString = @"EXPERIMENT_DONE";
	const int experimentEndSearchRadius = [experimentEndString length];
	NSRange searchRange = NSMakeRange(MAX(oldLength-experimentEndSearchRadius, 0), [evaluatorOutput length] - oldLength + experimentEndSearchRadius);
	if(searchRange.location + searchRange.length >= newLength)
		searchRange.length = newLength - searchRange.location;
	NSRange endRange = [evaluatorOutput rangeOfString:experimentEndString options:NSLiteralSearch range:searchRange];
	if(startRange.length > 0 && endRange.length > 0){
		// Stop observing and release the task when the evaluation is
		// complete.
		[[NSNotificationCenter defaultCenter] removeObserver:self];
		evaluator = nil;

		[evaluatorOutput setString:[evaluatorOutput substringWithRange:NSMakeRange(startRange.location+startRange.length, endRange.location+endRange.length-startRange.location-startRange.length)]]; //startRange.location+startRange.length] substringToIndex:endRange.location+endRange.length]];
		//NSLog(evaluatorOutput);
		[self finishCalculation];
		
		 // note that we don't do [evaluatorOutput setString:@""],
		evaluatorOutput = nil;     // because other objects may still need this string
		
		[self setCalculationInProgress:NO];

		return;
	}
	
	[fromEvaluator readInBackgroundAndNotify];
}

- (void)finishCalculation
{
	// finishCalculation will only be called after the evaluatorOutput
	// string is completed; calling setPlotCode strings will set
	// generatedPlotCode and fixedPlotCode.
	[self setPlotCodeStrings];
	
	// spawn the plot window with the data from this run
	[self addWindowController:plotWindowController];
	[[plotWindowController window] setTitle:[self titleForPlotWindow]];
	
	[plotWindowController setPlotCode:
		@[evaluatorOutput, generatedPlotCode, fixedPlotCode]];
	
	[plotWindowController selectPlot:nil];
	hasPlotData = YES;
	[plotWindowController showWindow:self];
	
	// release generatedPlotCode
	generatedPlotCode = nil;
	
	// reregister for notifications. I'm not sure *why* this is necessary, but it is
	[self registerForNotifications];
}

- (void)setCalculationInProgress:(BOOL)calcInProgress
{
	calculationInProgress = calcInProgress;
	
	if(calculationInProgress) {
		[calculateButton setTitle:@"Stop"];
		[calculateButton setKeyEquivalent:@"\e"];
		[progressIndicator setHidden:NO];
		[progressIndicator startAnimation:self];
	} else {
		[calculateButton setTitle:@"Calculate"];
		[calculateButton setKeyEquivalent:@"\r"];
		[progressIndicator stopAnimation:self];
		[progressIndicator setHidden:YES];
	}
}

- (NSArray*)trials
{
	return trials;
}

- (void)printDocumentWithSettings:(NSDictionary *)printSettings showPrintPanel:(BOOL)showPrintPanel delegate:(id)delegate didPrintSelector:(SEL)didPrintSelector contextInfo:(void *)contextInfo
{
	if(hasPlotData){
		[[plotWindowController plotView] print:nil];
	}
	else{
		[[NSAlert alertWithMessageText:@"There is nothing to print."
			defaultButton:@"OK" alternateButton:nil otherButton:nil
			informativeTextWithFormat:@"You must create a plot for this document in order to print."]
			beginSheetModalForWindow:[self windowForSheet] modalDelegate:nil didEndSelector:nil contextInfo:nil];
	}
}

- (void)close
{
	NSArray *windowControllers = [self windowControllers];
	int j;
	for(j = 0; j < [windowControllers count]; j++){
		NSWindowController *controller = windowControllers[j];
		if(controller)
			[controller close];
	}
	[super close];
}

- (void)setFileName:(NSString*)fileName
{
	[super setFileName:fileName];
	if(plotWindowController){
		[[plotWindowController window] setTitle:[self titleForPlotWindow]];
		[[plotWindowController window] display];
	}
}

- (void)makeWindowControllers
{
	parametersWindowController = [[NSWindowController alloc] initWithWindowNibName:[self windowNibName] owner:self];
	[parametersWindowController setShouldCloseDocument:YES];
	
	plotWindowController = [[PlotWindowController alloc] initWithWindowNibName:@"Plot"];
	[self addWindowController:parametersWindowController];
	[self addWindowController:plotWindowController];
}

- (void)showWindows
{
	if(parametersWindowController)
		[parametersWindowController showWindow:self];
	if(plotWindowController && hasPlotData)
		[plotWindowController showWindow:self];
}

- (void)windowControllerDidLoadNib:(NSWindowController *) aController
{
    [super windowControllerDidLoadNib:aController];
	[self synchronizeUserInterfaceToParams];
}

- (IBAction)synchronizeParametersAndUserInterface:(id)sender
{
	// First make sure that we read the new UI settings into
	// the parameters.
	[self synchronizeParamsToUserInterface];
	// Second, since we modify some parts of the UI based on
	// settings controlled by *other* parts of the UI,
	// resynch the user interface to match the parameters.
	[self synchronizeUserInterfaceToParams];
}

- (BOOL)textShouldBeginEditing:(NSText *)textObject
{
	return NO;
}

- (void)synchronizeParamsToUserInterface;
{
	NSAssert(0, @"Subclasses of PopBioDocument must override synchronizeParamsToUserInterface.");
}

- (void)synchronizeUserInterfaceToParams;
{
	NSAssert(0, @"Subclasses of PopBioDocument must override synchronizeUserInterfaceToParams.");
}

- (NSArray*)argumentsForEvaluator;
{
	NSAssert(0, @"Subclasses of PopBioDocument must override argumentsForEvaluator.");
	return nil;
}

- (NSString*)parametersReport;
{
	NSAssert(0, @"Subclasses of PopBioDocument must override parametersReport.");
	return nil;
}

- (void)setPlotCodeStrings
{
	NSAssert(0, @"Subclasses of PopBioDocument must override setPlotCodeStrings.");
}

- (NSString*)titleForPlotWindow;
{
	NSAssert(0, @"Subclasses of PopBioDocument must override titleForPlotWindow.");
	return nil;
}

- (NSString *)windowNibName
{
	NSAssert(0, @"Subclasses of PopBioDocument must override windowNibName");
	return nil;
}

- (IBAction)helpButton:(id)sender
{
	NSAssert(0, @"Subclasses of PopBioDocument must override helpButton:");
}

// Subclasses of PopBioDocument may override registerForNotifications
- (void)registerForNotifications { return; } 

@end
