//
//  PlotViewZoomTool.m
//  PopBio
//
//  Created by Stanislav Kounitski on 10/17/07.
//  This work is provided under the terms of the Educational Community License 1.0, a copy of which is included with the source code.
//

#import "PlotViewZoomTool.h"
#import "PlotView.h"
#import "RectWrapper.h"
#import "PlotViewZoomAnimation.h"

@interface PlotViewZoomTool() <NSAnimationDelegate>
@end

@interface PlotViewZoomTool(Private)
- (NSRect)_rectBetweenPoint:(NSPoint)a andPoint:(NSPoint)b;
@end

@implementation PlotViewZoomTool(Private)

- (NSRect)_rectBetweenPoint:(NSPoint)a andPoint:(NSPoint)b
{
	double ox = MIN(a.x, b.x);
	double oy = MIN(a.y, b.y);
	return NSMakeRect( ox,oy, fabs(a.x-b.x), fabs(a.y-b.y));
}
@end

@implementation PlotViewZoomTool

- (id)init
{
	self = [super init];
    if (self) {
        _viewRectStack = [[NSMutableArray alloc] init];
        _animateZooming = YES;
        _mouseIsBeingDragged = NO;
        _mouseControlsDisabled = NO;
    }
	return self;
}

- (void)zoomToRect:(NSRect)targetRect
{
	[self graphicallyZoomToRect:targetRect];
	[_viewRectStack addObject:[RectWrapper wrapperForRect:targetRect]];
	[_unzoomButton setEnabled:YES];
	[_zoomOutButton setEnabled:YES];
}

- (void)graphicallyZoomToRect:(NSRect)targetRect
{
	if(_animateZooming) {
		_mouseControlsDisabled = YES;
		[_plotView setTicksEnabled:NO];
		
		PlotViewZoomAnimation *zoomAnimation = [[PlotViewZoomAnimation alloc] initWithDuration:0.5
			animationCurve:NSAnimationEaseOut plotView:_plotView
			initialRect:[[_viewRectStack lastObject] rect] finalRect:targetRect];
		
        [zoomAnimation setFrameRate:30.0];
        [zoomAnimation setAnimationBlockingMode:NSAnimationNonblocking];
        [zoomAnimation setDelegate:self];
		[zoomAnimation startAnimation];
	}
	else {
		[_plotView setZoomRectEnabled:NO];
		[_plotView setViewRect:targetRect];
		[_plotView setNeedsDisplay:YES];
	}
}

- (void)animationDidEnd:(NSAnimation*)animation
{
	[_plotView setTicksEnabled:YES];
	[_plotView setZoomRectEnabled:NO];
	[_plotView setNeedsDisplay:YES];
	_mouseControlsDisabled = NO;
}

// -------------------------------------------------
// functions implementing the buttons on the toolbar
// -------------------------------------------------

- (IBAction)unzoom:(id)sender;
{
	NSRect dataRect = [_plotView dataRect];
	[self graphicallyZoomToRect:dataRect];
	[_viewRectStack removeAllObjects];
	[_viewRectStack addObject:[RectWrapper wrapperForRect:dataRect]];
	
	[_unzoomButton setEnabled:NO];
	[_zoomOutButton setEnabled:NO];
}

- (IBAction)zoomOut:(id)sender
{
	if([_viewRectStack count] == 2)
		[self unzoom:sender];
	else if([_viewRectStack count] > 2) {
		NSRect zoomRect = [_viewRectStack[[_viewRectStack count]-2] rect];
		[self graphicallyZoomToRect:zoomRect];
		[_viewRectStack removeLastObject];
	}
}

- (IBAction)doubleXAxis:(id)sender
{
	NSRect zoomRect = [[_viewRectStack lastObject] rect];
	zoomRect.size.width *= 2.0;
	[self zoomToRect:zoomRect];
}

- (IBAction)halfXAxis:(id)sender
{
	NSRect zoomRect = [[_viewRectStack lastObject] rect];
	zoomRect.size.width /= 2.0;
	[self zoomToRect:zoomRect];
}

- (IBAction)doubleYAxis:(id)sender
{
	NSRect zoomRect = [[_viewRectStack lastObject] rect];
	zoomRect.size.height *= 2.0;
	[self zoomToRect:zoomRect];
}

- (IBAction)halfYAxis:(id)sender
{
	NSRect zoomRect = [[_viewRectStack lastObject] rect];
	zoomRect.size.height /= 2.0;
	[self zoomToRect:zoomRect];
}

@end

// ----------------------------
// functions for event handling
// ----------------------------

@implementation PlotViewZoomTool (PlotViewMouseDelegate)

- (NSCursor*)cursorForPlotView
{
	return [NSCursor crosshairCursor];
}

- (void) mouseDownAtPoint:(NSPoint)pixelPoint
{
	if(!_mouseControlsDisabled)
		_zoomRectStartPoint = pixelPoint;
}

- (void) mouseDraggedAtPoint:(NSPoint)pixelPoint
{
	if(!_mouseControlsDisabled) {
		_mouseIsBeingDragged = YES;
		_zoomRectEndPoint = pixelPoint;
		_zoomRect = [self _rectBetweenPoint:_zoomRectStartPoint andPoint:_zoomRectEndPoint];
		NSRect zoomDataRect = [_plotView dataRectFromPixelRect:_zoomRect];
		zoomDataRect = NSIntersectionRect(zoomDataRect, [_plotView viewRect]);
		[_plotView setZoomRectEnabled:YES];
		[_plotView setZoomRect:zoomDataRect];
		[_plotView setNeedsDisplay:YES];
	}
}

- (void) mouseUpAtPoint:(NSPoint)pixelPoint
{
	if (!_mouseControlsDisabled && _mouseIsBeingDragged) {
		_zoomRectEndPoint = pixelPoint;
		_zoomRect = [self _rectBetweenPoint:_zoomRectStartPoint andPoint:_zoomRectEndPoint];
		NSRect zoomDataRect = [_plotView dataRectFromPixelRect:_zoomRect];
		zoomDataRect = NSIntersectionRect(zoomDataRect, [_plotView viewRect]);
		_mouseIsBeingDragged = NO;
		
		if(zoomDataRect.size.width > 0.0 && zoomDataRect.size.height > 0.0)
			[self zoomToRect:zoomDataRect];
		else
			[_plotView setZoomRectEnabled:NO];
	}
}

- (void)dataRectHasChanged
{
	_animateZooming = NO;
	[self unzoom:self];
	_animateZooming = YES;
}

@end
