//
//  PlotViewInteractivity.m
//  PopBio
//
//  Created by Evan Jones on 4/2/07.
//  Last modified by Stanislav Kounitski on 8/16/07
//  This work is provided under the terms of the Educational Community License 1.0, a copy of which is included with the source code.
//

#import "PlotView.h"
#import "RectWrapper.h"

@implementation PlotView (PlotViewInteractivity)

// -------------------------------------------------
// functions implementing the buttons on the toolbar
// -------------------------------------------------

- (IBAction)toggleDisplayGrid:(id)sender
{
	_displayGrid = !_displayGrid;
	[self setNeedsDisplay:YES];
}

- (IBAction)toggleLogLinearYAxis:(id)sender
{
	_yAxisIsLogarithmic = !_yAxisIsLogarithmic;
	[self setNeedsDisplay:YES];
}

// ----------------------------
// functions for event handling
// ----------------------------

#define WHERE [self convertPoint:[theEvent locationInWindow] fromView:nil]

- (void) mouseDown:(NSEvent *)theEvent 
{
	if(mouseDelegate != nil && [mouseDelegate respondsToSelector:@selector(mouseDownAtPoint:)])
		[mouseDelegate mouseDownAtPoint:WHERE];
}

- (void) mouseUp:(NSEvent *)theEvent 
{
	if(mouseDelegate != nil && [mouseDelegate respondsToSelector:@selector(mouseUpAtPoint:)])
		[mouseDelegate mouseUpAtPoint:WHERE];
}

- (void) mouseDragged:(NSEvent *) theEvent 
{
	if(mouseDelegate != nil && [mouseDelegate respondsToSelector:@selector(mouseDraggedAtPoint:)])
		[mouseDelegate mouseDraggedAtPoint:WHERE];
}

// -----------------------------------------
// functions for saving data and preferences
// -----------------------------------------

- (void)saveObject:(id)object asFile:(NSString*)fileType withDefaultFilename:(NSString*)defaultFilename
{
	NSSavePanel *sp = [NSSavePanel savePanel];
    [sp setAllowedFileTypes:@[fileType]];
    [sp setDirectoryURL:nil];
    [sp setNameFieldStringValue:defaultFilename];
    int runResult = [sp runModal];
	if (runResult == NSOKButton) {
		if (![object writeToURL:[sp URL] atomically:YES]){
			[[NSAlert alertWithMessageText:@"An error occurred while saving."
				defaultButton:@"OK" alternateButton:nil otherButton:nil
				informativeTextWithFormat:@"An error occurred while attempting to save the parameters."]
				beginSheetModalForWindow:[self window] modalDelegate:nil didEndSelector:nil contextInfo:nil];
		}
	}
	
	[saveMenuPopUpButton selectItemAtIndex:0];
}

- (IBAction)saveParametersAsText:(id)sender;
{
	NSString *outString = _plot[@"parameters-report"];
	NSString *defaultFilename = [NSString stringWithFormat:@"%@ Parameters.txt",_plot[@"long-title"]];
	[self saveObject:outString asFile:@"txt" withDefaultFilename:defaultFilename];
}

- (IBAction)saveResultsAsText:(id)sender;
{
	NSString *outString = _plot[@"results-report"];
	NSString *defaultFilename = [NSString stringWithFormat:@"%@ Results.txt",_plot[@"long-title"]];
	[self saveObject:outString asFile:@"txt" withDefaultFilename:defaultFilename];
}

- (IBAction)savePlotAsTIFF:(id)sender;
{
	NSData *outData = [[[NSImage alloc] initWithData:[self dataWithEPSInsideRect:_labelRect]] TIFFRepresentation];
	NSString *defaultFilename = [NSString stringWithFormat:@"%@ Plot.tif",_plotTitle];
	[self saveObject:outData asFile:@"tif" withDefaultFilename:defaultFilename];
}

- (IBAction)savePlotAsPDF:(id)sender;
{
	NSData *outData = [self dataWithPDFInsideRect:_labelRect];
	NSString *defaultFilename = [NSString stringWithFormat:@"%@ Plot.pdf",_plotTitle];
	[self saveObject:outData asFile:@"pdf" withDefaultFilename:defaultFilename];
}

// --------------------------------------------
// functions for editing the drawing parameters
// --------------------------------------------

- (IBAction)editDrawingParameters:(id)sender
{
	if(!parametersPanel)
		[NSBundle loadNibNamed:@"PlotParametersPanel" owner:self];
		
	[plotTitleTextField setStringValue:[self _plotTitle]];
	[xAxisLabelTextField setStringValue:[self _xAxisLabel]];
	[minimumXTextField setDoubleValue:_dataRect.origin.x];
	[maximumXTextField setDoubleValue:_dataRect.origin.x+_dataRect.size.width];
	[xTicIncrementTextField setDoubleValue:_dataRect.size.width/(double)_maxXLabels];
	[yAxisLabelTextField setStringValue:[self _yAxisLabel]];
	[minimumYTextField setDoubleValue:_dataRect.origin.y];
	[maximumYTextField setDoubleValue:_dataRect.origin.y+_dataRect.size.height];
	[yTicIncrementTextField setDoubleValue:_dataRect.size.height/(double)_maxYLabels];
		
	[NSApp beginSheet:parametersPanel
		modalForWindow:[self window]
		modalDelegate:nil
		didEndSelector:nil
		contextInfo:nil];
		
	[NSApp runModalForWindow:parametersPanel];
	
	[NSApp endSheet:parametersPanel];
	[parametersPanel orderOut:self];
}

- (IBAction)closeParametersPanelOK:(id)sender
{
	[_plotTitle setString:[plotTitleTextField stringValue]];
	[_xAxisLabel setString:[xAxisLabelTextField stringValue]];
	_dataRect.origin.x = [minimumXTextField doubleValue];
	_dataRect.size.width = [maximumXTextField doubleValue] - _dataRect.origin.x;
	
	if([xTicIncrementTextField doubleValue])
		_maxXLabels = _dataRect.size.width / [xTicIncrementTextField doubleValue];
	else
		_maxXLabels = 0;
	
	[_yAxisLabel setString:[yAxisLabelTextField stringValue]];
	_dataRect.origin.y = [minimumYTextField doubleValue];
	_dataRect.size.height = [maximumYTextField doubleValue] - _dataRect.origin.y;
	
	if([yTicIncrementTextField doubleValue])
		_maxYLabels = _dataRect.size.height / [yTicIncrementTextField doubleValue];
	else
		_maxYLabels = 0;
		
	[self dataRectHasChanged];

	[NSApp stopModal];
}

- (IBAction)closeParametersPanelCancel:(id)sender
{
	[NSApp stopModal];
}

// -------------------------------------------------------------
// functions for setting the plot and adding and removing curves
// -------------------------------------------------------------

- (void)addCurve:(NSDictionary*)aCurve;
{
	NSMutableArray *_curves = _plot[@"curves"];
	if ([_curves containsObject:aCurve] == NO)
		[_curves addObject:aCurve];
	[self makePlotValuesAgree];
	[self setNeedsDisplay:YES];
}

- (void)removeCurve:(NSDictionary*)aCurve;
{
	[_plot[@"curves"] removeObject:aCurve];
	[self makePlotValuesAgree];
	[self setNeedsDisplay:YES];
}

- (void)displayPlot:(NSDictionary*)aPlot;
{
	if(_plot == aPlot)
		return;
	_plot = aPlot;
	[self makePlotValuesAgree];
	[self setNeedsDisplay:YES];
}

@end