//
//  PlotViewDrawing.m
//  PopBio
//
//  Created by Evan Jones on 4/2/07.
//  Last modified by Stanislav Kounitski on 8/16/07
//  This work is provided under the terms of the Educational Community License 1.0, a copy of which is included with the source code.
//

#import "PlotView.h"
#import "RectWrapper.h"

@implementation PlotView (PlotViewDrawing)

- (void)drawRect:(NSRect)rect;
{
	 // Divide up the view into different rects:
	_labelRect = rect;
	_plotRect = NSInsetRect(_labelRect, kHorizontalTextAreaSize*2, kVerticalTextAreaSize*2);
	_xLabelRect = NSMakeRect( _labelRect.origin.x + kHorizontalTextAreaSize*2, _labelRect.origin.y, 
							  NSWidth(_plotRect), NSMinY(_plotRect)-NSMinY(_labelRect));
	_yLabelRect = NSMakeRect( _labelRect.origin.x, _labelRect.origin.y+2*kVerticalTextAreaSize,
							  NSMinX(_plotRect)-NSMinX(_labelRect), NSHeight(_plotRect));

	[[NSColor whiteColor] set];
	NSRectFill(rect);
	[[NSColor blackColor] set];
	
	// Draw axes
	[[self axesPath] stroke];
	
	if(_drawTicks) {
		// calculate a valid number of ticks
		// This must happen before any labels, ticks or grid are drawn
		[self setLabelCount];
	
		// set ticks, grid, and labels
		[self setTicks];
	
		// draw grid
		if (_displayGrid){
			[[NSColor gridColor] set];
			[[self gridPath] stroke];
			[[NSColor blackColor] set];
		}
	
		// draw ticks
		if(_xTicks != NULL && [_xTicks count] > 0) {
			[[self xTicksPath] stroke];
			[self writeHorizontalLabels];
		}
		if(_yTicks != NULL && [_yTicks count] > 0) {
			[[self yTicksPath] stroke];
			[self writeVerticalLabels];
		}
	}
		
	// draw headers
	[self writeHeadings];
	
	// if we're in the act of zooming, draw the rubber band
	if ( _drawZoomRect){
		[[NSColor redColor] set];
		NSFrameRect([self pixelRectFromDataRect:_zoomRect]);
		[[NSColor blackColor] set];
	}
	
	// Clip to _plotRect now that all drawing outside of it has been done.
	// Make the clipRect one pixel larger than _plotRect to make sure that border conditions are recorded
	[NSBezierPath clipRect:NSMakeRect(NSMinX(_plotRect)-1, NSMinY(_plotRect)-1, NSWidth(_plotRect)+2, NSHeight(_plotRect)+2)];
	
	// Draw plots
	NSArray *_curves = _plot[@"curves"];
	
	int i;
	for(i=0; i<[_curves count]; i++) {
		NSDictionary *curve = _curves[i];
		NSString *specialCurve = curve[@"special-curve"];
		if(!specialCurve) {
			[[self colorForCurveAtIndex:i] set];
			[[self pathForCurve:curve] stroke];
		}
		else if([specialCurve isEqualToString:@"line"]) {
			[[self colorForCurveAtIndex:i] set];
			[[self pathForLine:curve] stroke];
		}
		else if([specialCurve isEqualToString:@"vLine"]) {
			[[self colorForCurveAtIndex:i] set];
			[[self pathForVerticalLine:curve] stroke];
		}
		else if([specialCurve isEqualToString:@"parabola"]) {
			[[self colorForCurveAtIndex:i] set];
			[[self pathForParabola:curve] stroke];
		}
		else if([specialCurve isEqualToString:@"Holling II isocline"]) {
			[[self colorForCurveAtIndex:i] set];
			[[self pathForHolling2Isocline:curve] stroke];
		}
		else if([specialCurve isEqualToString:@"image"]) {
			NSImage *img = curve[@"image"];
			NSPoint dataPoint = NSMakePoint([curve[@"x"] doubleValue], [curve[@"y"] doubleValue]);
			NSPoint pixelPoint = [self pixelPointFromDataPoint:dataPoint];
			pixelPoint.x = round(pixelPoint.x - [curve[@"x-offset"] doubleValue]);
			pixelPoint.y = round(pixelPoint.y - [curve[@"y-offset"] doubleValue]);
			double opacity = [curve[@"opacity"] doubleValue];
			[img drawAtPoint:pixelPoint fromRect:NSZeroRect operation:NSCompositeSourceOver fraction:opacity];
		}
	}
	
	[self resetCursorRects];
}

-(NSBezierPath*)pathForVerticalLine:(NSDictionary*)vLine
{
	NSBezierPath *dataPath = [NSBezierPath bezierPath];
	double x = [vLine[@"x"] doubleValue];
	double y_min = NSMinY(_viewRect), y_max = NSMaxY(_viewRect);
	
	[dataPath moveToPoint: [self pixelPointFromDataPoint: NSMakePoint(x, y_min)]];
	[dataPath lineToPoint: [self pixelPointFromDataPoint: NSMakePoint(x, y_max)]];
	
	return dataPath;
}

-(NSBezierPath*)pathForLine:(NSDictionary*)line
{
	NSBezierPath *dataPath = [NSBezierPath bezierPath];
	double m = [line[@"slope"] doubleValue];
	double b = [line[@"y-intercept"] doubleValue];
	double x_min = NSMinX(_viewRect), x_max = NSMaxX(_viewRect);
	double x, dx = (x_max - x_min) / 100.0;
	
	[dataPath moveToPoint: [self pixelPointFromDataPoint: NSMakePoint(x_min, m*x_min + b)]];
	for(x = x_min+dx; x <= x_max; x += dx)
		[dataPath lineToPoint:[self pixelPointFromDataPoint:NSMakePoint(x, m*x + b)]];
	
	return dataPath;
}

-(NSBezierPath*)pathForParabola:(NSDictionary*)parabola
{
	NSBezierPath *dataPath = [NSBezierPath bezierPath];
	double a = [parabola[@"a"] doubleValue];
	double b = [parabola[@"b"] doubleValue];
	double c = [parabola[@"c"] doubleValue];
	double x_min = NSMinX(_viewRect), x_max = NSMaxX(_viewRect);
	double x, dx = (x_max - x_min) / 100.0;
	
	x = x_min;
	[dataPath moveToPoint: [self pixelPointFromDataPoint: NSMakePoint(x, a*x*x + b*x + c)]];
	for(x = x_min+dx; x <= x_max; x += dx)
		[dataPath lineToPoint:[self pixelPointFromDataPoint:NSMakePoint(x, a*x*x + b*x + c)]];
	
	return dataPath;
}

-(NSBezierPath*)pathForHolling2Isocline:(NSDictionary*)h2iso
{
	NSBezierPath *dataPath = [NSBezierPath bezierPath];
	double K = [h2iso[@"K"] doubleValue];
	double a = [h2iso[@"a"] doubleValue];
	double m = [h2iso[@"m"] doubleValue];
	double w = [h2iso[@"w"] doubleValue];
	double x_min = NSMinX(_viewRect), x_max = NSMaxX(_viewRect);
	double x, dx = (x_max - x_min) / 100.0;
	
	x = x_min;
	[dataPath moveToPoint: [self pixelPointFromDataPoint: NSMakePoint(x, x/((K*m/a)/(K-x)-w))]];
	for(x = x_min+dx; x <= x_max; x += dx)
		[dataPath lineToPoint:[self pixelPointFromDataPoint:NSMakePoint(x, x/((K*m/a)/(K-x)-w))]];
	
	return dataPath;
}

-(NSBezierPath*)pathForCurve:(NSDictionary*)curve
{
	NSBezierPath *dataPath = [NSBezierPath bezierPath];
	NSArray *xData = curve[@"x-data"];
	NSArray *yData = curve[@"y-data"];

    // If the data are malformed then there is no path.
    if (![xData isKindOfClass:[NSArray class]] || ![yData isKindOfClass:[NSArray class]]) {
        return nil;
    }

	NSPoint dataPoint, pixelPoint;
	int i = 0;
	
	dataPoint = NSMakePoint([xData[i] doubleValue], [yData[i] doubleValue]);
	pixelPoint = [self pixelPointFromDataPoint:dataPoint];
	[dataPath moveToPoint:pixelPoint];
	
	for( i = 1; i < MIN([xData count], [yData count]); i += 1 ){
		dataPoint = NSMakePoint([xData[i] doubleValue], [yData[i] doubleValue]);
		pixelPoint = [self pixelPointFromDataPoint:dataPoint];
		// We need to clamp these values becuase the program will actually crash if we try
		// to call [NSBezierPath stroke] with an INF point in the path.
		if(pixelPoint.x > PLOT_MAX)
			pixelPoint.x = PLOT_MAX;
		if(pixelPoint.y > PLOT_MAX)
			pixelPoint.y = PLOT_MAX;
			
		[dataPath lineToPoint:pixelPoint];
	}
	
	return dataPath;
}

-(NSBezierPath *)axesPath;
{
	NSBezierPath *axesPath = [NSBezierPath bezierPath];
	[axesPath moveToPoint:_plotRect.origin];
	[axesPath relativeLineToPoint:NSMakePoint(0,NSHeight(_plotRect))];
	[axesPath moveToPoint:_plotRect.origin];
	[axesPath relativeLineToPoint:NSMakePoint(NSWidth(_plotRect),0)];
	return axesPath;
}

-(NSBezierPath *)gridPath;
{
	NSBezierPath *gridPath = [NSBezierPath bezierPath];

	for(int j = ([_xTicks[0] doubleValue] == 0 ? 1 : 0); j < [_xTicks count]; j++) {
		double x = _plotRect.origin.x + [_xTicks[j] doubleValue];
		[gridPath moveToPoint:NSMakePoint(x, _plotRect.origin.y)];
		[gridPath relativeLineToPoint:NSMakePoint(0, NSHeight(_plotRect))];
	}
	
	for(int j = ([_yTicks[0] doubleValue] == 0 ? 1 : 0); j < [_yTicks count]; j++) {
		double y = _plotRect.origin.y + [_yTicks[j] doubleValue];
		[gridPath moveToPoint:NSMakePoint( _plotRect.origin.x, y)];
		[gridPath relativeLineToPoint:NSMakePoint( NSWidth(_plotRect), 0)];
	}

	return gridPath;
}

- (NSBezierPath*)xTicksPath;
{
	NSBezierPath *tickPath = [NSBezierPath bezierPath];
	double y = _plotRect.origin.y;
	
	for(int j = ([_xTicks[0] doubleValue] == 0 ? 1 : 0); j < [_xTicks count]; j++) {
		double x = [_xTicks[j] doubleValue];
		[tickPath moveToPoint:NSMakePoint(_plotRect.origin.x + x,y)];
		[tickPath relativeLineToPoint:NSMakePoint(0,_tickLength)];
	}
	return tickPath;
}

- (NSBezierPath*)yTicksPath;
{
	NSBezierPath *tickPath = [NSBezierPath bezierPath];
	double x = _plotRect.origin.x;
	
	for(int j = ([_yTicks[0] doubleValue] == 0 ? 1 : 0); j < [_yTicks count]; j++) {
		double y = [_yTicks[j] doubleValue];
		[tickPath moveToPoint:NSMakePoint(x,_plotRect.origin.y + y)];
		[tickPath relativeLineToPoint:NSMakePoint(_tickLength,0)];
	}
	return tickPath;
}

-(void)writeHorizontalLabels;
{
	int j;
	for(j = 0; j < [_xTicks count]; j++) {
		double x = [_xTicks[j] doubleValue];
		NSString *labelString = _xLabels[j];
		NSRect boundingRect = [labelString boundingRectWithSize:NSMakeSize(100,100) options:0 attributes:_littleTextAttributes];
		NSRect drawingRect = NSOffsetRect( boundingRect, 
										   _xLabelRect.origin.x + x - NSWidth(boundingRect)/2, 
										   _xLabelRect.origin.y + NSHeight(_xLabelRect) - NSHeight(boundingRect)+2 );
		[labelString drawInRect:drawingRect withAttributes:_littleTextAttributes];
	}
}

-(void)writeVerticalLabels;
{
	int j;
	for(j = 0; j < [_yTicks count]; j++) {
		double y = [_yTicks[j] doubleValue];
		NSString *labelString = _yLabels[j];
		NSRect boundingRect = [labelString boundingRectWithSize:NSMakeSize(100,100) options:0 attributes:_littleTextAttributes];
		NSRect drawingRect = NSOffsetRect(boundingRect,
										  _yLabelRect.origin.x + NSWidth(_yLabelRect) - NSWidth(boundingRect) - 4, 
										  _yLabelRect.origin.y + y - NSHeight(boundingRect)/4);
		[labelString drawInRect:drawingRect withAttributes:_littleTextAttributes];
	}
}

-(void)writeHeadings;
{	
	// Put the plot label in Helvetica 12, at the top center of the plot.
	NSRect boundingRect = [[self _plotTitle] boundingRectWithSize:NSMakeSize(1000,1000) options:0 attributes:_bigTextAttributes];
	[[self _plotTitle] drawInRect:
		NSMakeRect([self frame].size.width/2.0-boundingRect.size.width/2.0,
				   [self frame].size.height-boundingRect.size.height, boundingRect.size.width, boundingRect.size.height) withAttributes:_bigTextAttributes];
	
	// Put the x-axis label in Helvetica 10 (same as tick labels) in the bottom center.
	boundingRect = [[self _xAxisLabel] boundingRectWithSize:NSMakeSize(1000,1000) options:0 attributes:_littleTextAttributes];
	[[self _xAxisLabel] drawInRect:
		NSMakeRect([self frame].size.width/2.0-boundingRect.size.width/2.0,
				   kVerticalTextAreaSize-boundingRect.size.height, boundingRect.size.width, boundingRect.size.height) withAttributes:_littleTextAttributes];
	
	// Put the y-axis label in Helvetica 10 (same as tick labels) on the left side, just below the plot label.
	boundingRect = [[self _yAxisLabel] boundingRectWithSize:NSMakeSize(1000,1000) options:0 attributes:_littleTextAttributes];
	NSAffineTransform *xform = [NSAffineTransform transform];
	[xform translateXBy:kHorizontalTextAreaSize-boundingRect.size.height yBy:[self frame].size.height/2.0-boundingRect.size.width/2.0];
	[xform rotateByDegrees:90.0];
	[xform concat];
	[[self _yAxisLabel] drawInRect:
		NSMakeRect(0.0, 0.0, boundingRect.size.width, boundingRect.size.height) withAttributes:_littleTextAttributes];
	[xform invert];
	[xform concat];	
}

@end