//
//  PlotView.m
//  PopBio
//
//  Created by Evan Jones on 4/2/07.
//  Last modified by Stanislav Kounitski on 8/16/07
//  This work is provided under the terms of the Educational Community License 1.0, a copy of which is included with the source code.
//

#import "PlotView.h"
#import "RectWrapper.h"
#import <math.h>

// -------------------------------------------------------------------------------
// auxiliary math functions
// -------------------------------------------------------------------------------

@interface PlotView(Private)
- (double)floor:(double)value withSpacing:(double)spacing;
- (double)ceiling:(double)value withSpacing:(double)spacing;
- (double)round:(double)value withSpacing:(double)spacing;
@end

@implementation PlotView(Private)

- (double)floor:(double)value withSpacing:(double)spacing
{
	double result;
	if(spacing > 0.0)
		result = spacing * (double)floor(value/spacing);
	else
		result = 0.0;
		
	return result;
}

- (double)ceiling:(double)value withSpacing:(double)spacing
{
	double result;
	if(spacing > 0.0)
		result = spacing * (double)ceil(value/spacing);
	else
		result = 0.0;
		
	return result;
}

- (double)round:(double)value withSpacing:(double)spacing
{
	double lower = [self floor:value withSpacing:spacing];
	double upper = [self ceiling:value withSpacing:spacing];
	if(0.95*(value-lower) < upper-value) // slight preference for returning the lower value
		return lower;
	return upper;
}

@end

@implementation PlotView

// -------------------------------------------------------------------------------
// initialization and deallocation
// -------------------------------------------------------------------------------

-(id)initWithFrame:(NSRect)frame;
{
	self = [super initWithFrame:frame];
	if (self){
		_drawTicks = YES;
		_drawZoomRect = NO;		
		_displayGrid = NO;
		_yAxisIsLogarithmic = NO;
		
		_littleTextAttributes = @{NSFontAttributeName: [NSFont fontWithName:@"Helvetica" size:10]};
		_bigTextAttributes    = @{NSFontAttributeName: [NSFont fontWithName:@"Helvetica" size:12]};
		
		_tickLength = 5.0;
		
		_plotTitle = [NSMutableString stringWithString:@"plot"];
		_xAxisLabel = [NSMutableString stringWithString:@"x axis"];
		_yAxisLabel = [NSMutableString stringWithString:@"y axis"];
	}
	return self;
}


// -------------------------------------------------------------------------------
// basic methods
// -------------------------------------------------------------------------------

- (void)setZoomRectEnabled:(BOOL)b; { _drawZoomRect = b; }
- (void)setTicksEnabled:(BOOL)b; { _drawTicks = b; }
- (void)setViewRect:(NSRect)viewRect { _viewRect = viewRect; }
- (void)setZoomRect:(NSRect)zoomRect; {	_zoomRect = zoomRect; }
- (NSRect)viewRect { return _viewRect; }
- (NSRect)dataRect { return _dataRect; }
- (NSString*)_plotTitle { return _plotTitle; }
- (NSString*)_xAxisLabel { return _xAxisLabel; }
- (NSString*)_yAxisLabel { return _yAxisLabel; }

// ---------------------------------------------------------------------------
// the following functions are responsible for setting all of the parameters
// that PlotView needs to display correctly. makePlotValuesAgree sets the
// plot dimensions and the headings and axis labels, as well as specifying the
// maximum number of tick-marks on the axes; it should only be called when
// plots are added or taken away from the PlotView. setLabelCount determines
// the optimal number of labeled ticks on each axis, while setTicks specifies
// the coordinates and labels of each tick mark; these functions are called
// every time the user zooms.
// ---------------------------------------------------------------------------

-(NSRect)boundingRectOfCurve:(NSDictionary*)curve
{
	NSArray *xData = curve[@"x-data"];
	NSArray *yData = curve[@"y-data"];

    // If the data are malformed then the bounding rect is zero.
    if (![xData isKindOfClass:[NSArray class]] || ![yData isKindOfClass:[NSArray class]]) {
        return NSZeroRect;
    }

	int i, numPoints = MIN([xData count], [yData count]);   // Important! Sometimes, [xData count] > [yData count] (!?)
	double x, xMin = [xData[0] doubleValue], xMax=xMin;
	double y, yMin = [yData[0] doubleValue], yMax=yMin;
	
	for(i=1; i<numPoints; i++) {
		x = [xData[i] doubleValue];
		y = [yData[i] doubleValue];
		xMin = MIN(xMin, x);
		xMax = MAX(xMax, x);
		yMin = MIN(yMin, y);
		yMax = MAX(yMax, y);
	}
	
	NSNumber *xPadding = curve[@"x-padding"];
	NSNumber *yPadding = curve[@"y-padding"];
	if(xPadding)
		xMax += (xMax-xMin)*[xPadding doubleValue];
	if(yPadding)
		yMax += (yMax-yMin)*[yPadding doubleValue];
	
	return NSMakeRect(xMin, yMin, xMax-xMin, yMax-yMin);
}

-(NSRect)boundingRectOfAllCurves
{
	NSArray *_curves = _plot[@"curves"];
	NSRect boundingRect;
	NSDictionary *curve;
	BOOL rectInitialized=NO;
	int i;
	
	for(i=0; i < [_curves count]; i++) {
		curve = _curves[i];
		if(!curve[@"special-curve"]) {
			if(!rectInitialized) {
				rectInitialized = YES;
				boundingRect = [self boundingRectOfCurve:curve];
			}
			else {
				boundingRect = NSUnionRect(boundingRect, [self boundingRectOfCurve:curve]);
			}
		}
	}
	
	return (rectInitialized ? boundingRect : NSZeroRect);
}

-(NSRect)boundingRectOfPlot
{
	NSRect autoRect = [self boundingRectOfAllCurves];
	NSNumber *xMin = _plot[@"x-min"], *xMax = _plot[@"x-max"];
	NSNumber *yMin = _plot[@"y-min"], *yMax = _plot[@"y-max"];

	if(NSEqualRects(autoRect,NSZeroRect) && (!xMin || !xMax || !yMin || !yMax))
		return NSMakeRect(0,0,1,1);
	
	double x0=[xMin doubleValue], x1=[xMax doubleValue], y0=[yMin doubleValue], y1=[yMax doubleValue];
	if(!NSEqualRects(autoRect,NSZeroRect)) {
		double X0=NSMinX(autoRect), X1=NSMaxX(autoRect), Y0=NSMinY(autoRect), Y1=NSMaxY(autoRect);
		x0 = (!xMin ? X0 : MIN(x0, X0));
		x1 = (!xMax ? X1 : MAX(x1, X1));
		y0 = (!yMin ? Y0 : MIN(y0, Y0));
		y1 = (!yMax ? Y1 : MAX(y1, Y1));
	}
	NSRect finalRect = NSMakeRect(x0, y0, x1-x0, y1-y0);
	finalRect.size.width = MIN(finalRect.size.width, PLOT_MAX);
	finalRect.size.height = MIN(finalRect.size.height, PLOT_MAX);
	return finalRect;
}

-(void)makePlotLabelsAgree
{
	if(!_plot) {
		[_plotTitle setString:@"plot"];
		[_xAxisLabel setString:@"x axis"];
		[_yAxisLabel setString:@"y axis"];
		_maxXLabels = 10;
		_maxYLabels = 10;
	}
	else {
		_maxXLabels = [_plot[@"x-ticks"] intValue];
		_maxYLabels = [_plot[@"y-ticks"] intValue];
		[_plotTitle setString:_plot[@"long-title"]];
		[_xAxisLabel setString:_plot[@"x-axis-label"]];
		[_yAxisLabel setString:_plot[@"y-axis-label"]];
	}
}

-(void)makePlotValuesAgree;
{
	[self makePlotLabelsAgree];
	
	_dataRect = [self boundingRectOfPlot];
	double minY = NSMinY([self boundingRectOfAllCurves]);
	_logarithmicMinY = (minY < 1e-6 ? 1e-6 : pow(10,ceil(log10(minY))-1) );
	
	// _viewRect will change if we zoom
	// Note that by setting _viewRect to _dataRect here, we make
	// zooming non-persistent.  To make zooming persistent between
	// graphs, we'd have to put some extra logic in here.
	[self dataRectHasChanged];
}

- (void)setLabelCount;
{	
	// Figure out the on-screen size for labels along
	// the axes.  If two subsequent labels intersect, 
	// there are too many labels, so divide the number of
	// labels by half.

	BOOL xCollision = NO;
	BOOL yCollision;
	int j;
	_xIntervals = _maxXLabels;
	_yIntervals = _maxYLabels;
	
	do{
		xCollision = NO;
		yCollision = NO;
		NSRect lastRect;
		for(j = 0; j <= _xIntervals; j++){
			double x = NSMinX(_viewRect) + (double)j/_xIntervals *NSWidth(_viewRect);

			NSString *theString = [NSString stringWithFormat:@"%.2f", x];
			
			NSRect boundingRect = [theString boundingRectWithSize:NSMakeSize(100,100) options:0 attributes:_littleTextAttributes];
			NSRect drawingRect = NSOffsetRect( boundingRect, (double)j/_xIntervals*NSWidth(_xLabelRect),0);
			
			if(j > 0 && NSIntersectsRect(drawingRect, lastRect)){
				xCollision = YES;
				break;
			}
			lastRect = drawingRect;
		}
		
		for(j = 0; j <= _yIntervals; j++){
			double y = NSMinY(_viewRect) + (double)j/_yIntervals*NSHeight(_viewRect);
					
			NSString *theString = [NSString stringWithFormat:@"%.2f", y];
			
			NSRect boundingRect = [theString boundingRectWithSize:NSMakeSize(100,100) options:0 attributes:_littleTextAttributes];
			NSRect drawingRect = NSOffsetRect(boundingRect, 0, (double)j/_yIntervals*NSHeight(_yLabelRect));

			if(j > 0 && NSIntersectsRect(drawingRect, lastRect)){
				yCollision = YES;
				break;
			}
            lastRect = drawingRect;
		}
		
		if(xCollision)
			_xIntervals /= 2;
		if(yCollision)
			_yIntervals /= 2;
				
	}while(xCollision || yCollision);
}

- (void)setTicks
{
	if(_xIntervals == 0 || _yIntervals == 0) {
		_xTicks = NULL;
		_yTicks = NULL;
		_xLabels = NULL;
		_yLabels = NULL;
		return;
	}

	int j;
	if(NSWidth(_viewRect) > 0.0 && _xIntervals > 0.0){
		double dx = NSWidth(_viewRect) / _xIntervals;
		double Dx = [self round:dx withSpacing:pow(10,floor(log10(dx)))/2];
		double xMin = [self ceiling:NSMinX(_viewRect) withSpacing:Dx];
		double xMax = [self floor:NSMaxX(_viewRect) withSpacing:Dx];
		int numIntervals = (int)round( (xMax - xMin)/Dx );
		_xTicks = [NSMutableArray arrayWithCapacity: numIntervals+1];
		_xLabels = [NSMutableArray arrayWithCapacity: numIntervals+1];
		
		double dataX, plotX;
		for(j = 0; j <= numIntervals; j++) {
			dataX = xMin + j*Dx;
			plotX = [self pixelXFromDataX:dataX] - NSMinX(_plotRect);
			[_xTicks addObject:@(plotX)];
			[_xLabels addObject:[self makeLabelForValue:dataX withSpacing:Dx]];
		}
	}
	
	if(NSHeight(_viewRect) > 0.0 && _yIntervals > 0.0){
		// vertical ticks with logarithmic axis
		if(_yAxisIsLogarithmic) {
			double viewMinY = NSMinY(_viewRect);
			if(viewMinY < _logarithmicMinY)
				viewMinY = _logarithmicMinY;
			int minPowerOf10 = ceil(log10(viewMinY));
			int maxPowerOf10 = floor(log10(NSMaxY(_viewRect)));
			if(maxPowerOf10 < minPowerOf10) {
				_yTicks = NULL;
				_yLabels = NULL;
				return;
			}
			_yTicks = [NSMutableArray arrayWithCapacity: maxPowerOf10-minPowerOf10+1];
			_yLabels = [NSMutableArray arrayWithCapacity: maxPowerOf10-minPowerOf10+1];
			for(j = minPowerOf10; j <= maxPowerOf10; j++) {
				double dataY = pow(10,j);
				double plotY = [self pixelYFromDataY:dataY];
				[_yTicks addObject:@(plotY-NSMinY(_plotRect))];
				[_yLabels addObject:[NSString stringWithFormat:@"1e%d", j]];
				//the code below makes an attributed string, but I'm not sure what to do with it
				/*NSString *str = [NSString stringWithFormat:@"10%d", j];
				NSMutableAttributedString *label = [[NSMutableAttributedString alloc] initWithString:str];
				[label superscriptRange:NSMakeRange(1,[label length]-1)];
				[_yLabels addObject:label];*/
			}
			return;
		}
		
		double dy = NSHeight(_viewRect) / _yIntervals;
		double Dy;
		if(dy > 10000.0)
			Dy = [self round:dy withSpacing:pow(10,floor(log10(dy)))];
		else
			Dy = [self round:dy withSpacing:pow(10,floor(log10(dy)))/2];
		double yMin = [self ceiling:NSMinY(_viewRect) withSpacing:Dy];
		double yMax = [self floor:NSMaxY(_viewRect) withSpacing:Dy];
		int numIntervals = (int)round( (yMax - yMin)/Dy );
		_yTicks = [NSMutableArray arrayWithCapacity: numIntervals+1];
		_yLabels = [NSMutableArray arrayWithCapacity: numIntervals+1];
		
		double dataY, plotY;
		for(j = 0; j <= numIntervals; j++) {
			dataY = yMin + j*Dy;
			plotY = [self pixelYFromDataY:dataY] - NSMinY(_plotRect);
			[_yTicks addObject:@(plotY)];
			[_yLabels addObject:[self makeLabelForValue:dataY withSpacing:Dy]];
		}
	}
}

// *** OBSOLETE ***
-(void)setTicksOldskool
{
	int j;

	// horizontal ticks
	_xTicks = [NSMutableArray arrayWithCapacity: _xIntervals+1];
	_xLabels = [NSMutableArray arrayWithCapacity: _xIntervals+1];
	double dx = NSWidth(_viewRect) / _xIntervals;
	for(j = 0; j <= _xIntervals; j++) {
		double plotX = NSMinX(_plotRect) + (double)j / _xIntervals * NSWidth(_plotRect);
		double dataX = [self dataXFromPixelX:plotX];
		[_xTicks addObject:@(plotX-NSMinX(_plotRect))];
		[_xLabels addObject:[self makeLabelForValue:dataX withSpacing:dx]];
	}

	// vertical ticks
	_yTicks = [NSMutableArray arrayWithCapacity: _yIntervals+1];
	_yLabels = [NSMutableArray arrayWithCapacity: _yIntervals+1];
	double dy = NSHeight(_viewRect) / _yIntervals;
	for(j = 0; j <= _yIntervals; j++) {
		double plotY = NSMinY(_plotRect) + (double)j / _yIntervals * NSHeight(_plotRect);
		double dataY = [self dataYFromPixelY:plotY];
		[_yTicks addObject:@(plotY-NSMinY(_plotRect))];
		[_yLabels addObject:[self makeLabelForValue:dataY withSpacing:dy]];
	}
}

// -------------------------------------------------------------------------------
// makeLabelForValue and its auxiliary functions (floor/ceiling/round withSpacing)
// are used by setTicks to create string labels for each tickmark on the axes.
// -------------------------------------------------------------------------------

- (NSString *)makeLabelForValue:(double)value withSpacing:(double)spacing
{

	if(spacing > 1000.0) {
        NSMutableString *labelString;
		if(value == 0)
			return @"0";
	
		double number = spacing;
		int numZeroes = 0;
		while(number == 10*floor(number/10)) {
			number /= 10;
			numZeroes++;
		}
		int numSignificantDigits = floor(log10(value)) + 1 - numZeroes;
		
		if(numSignificantDigits <= 1)
			labelString = [NSMutableString stringWithFormat:@"%.0e", value];
		else if(numSignificantDigits == 2)
			labelString = [NSMutableString stringWithFormat:@"%.1e", value];
		else if(numSignificantDigits == 3)
			labelString = [NSMutableString stringWithFormat:@"%.2e", value];
		else
			labelString = [NSMutableString stringWithFormat:@"%.3e", value];
		
		[labelString replaceOccurrencesOfString:@"e+0" withString:@"e" options:0 range:NSMakeRange(0, [labelString length])];
		[labelString replaceOccurrencesOfString:@"e+" withString:@"e" options:0 range:NSMakeRange(0, [labelString length])];
		return labelString;
	}
	
	int numDigits = 0;
	spacing -= (double)floor(spacing);
	while(spacing > 0.0001) { // not 0.0 to correct for inexact representation of doubles in memory
		spacing = 10*spacing - (double)floor(10*spacing);
		numDigits++;
	}

    NSString *labelString;
	if(numDigits == 0)
		labelString = [NSString stringWithFormat:@"%d", (int)round(value)];
	else if(numDigits == 1)
		labelString = [NSString stringWithFormat:@"%.1f", value];
	else if(numDigits == 2)
		labelString = [NSString stringWithFormat:@"%.2f", value];
	else
		labelString = [NSString stringWithFormat:@"%.3f", value];
	
	return labelString;
}

- (void)dataRectHasChanged
{
	// this is a bit of a hack; ideally, we'd want this function to post a notification
	if(mouseDelegate != nil && [mouseDelegate respondsToSelector:@selector(dataRectHasChanged)])
		[mouseDelegate dataRectHasChanged];
}

// -----------------------------------------------------------
// functions for converting between data space and pixel space
// -----------------------------------------------------------

// data space to pixel space

-(double)pixelXFromDataX:(double)dataX;
{
	return NSMinX(_plotRect) + (dataX-NSMinX(_viewRect))/(NSWidth(_viewRect))*NSWidth(_plotRect);
}

-(double)pixelYFromDataY:(double)dataY;
{
	double viewMinY = NSMinY(_viewRect);
	double viewMaxY = NSMaxY(_viewRect);
	if(_yAxisIsLogarithmic) {
		if(viewMinY < _logarithmicMinY)
			viewMinY = _logarithmicMinY;
		dataY = log(dataY);
		viewMinY = log(viewMinY);
		viewMaxY = log(viewMaxY);
	}
	
	return NSMinY(_plotRect) + (dataY-viewMinY) / (viewMaxY-viewMinY) * NSHeight(_plotRect);
}

-(NSPoint)pixelPointFromDataPoint:(NSPoint)dataPoint;
{
	return NSMakePoint([self pixelXFromDataX:dataPoint.x],[self pixelYFromDataY:dataPoint.y]);
}

-(NSRect)pixelRectFromDataRect:(NSRect)dataRect;
{
	double xMin, xMax, yMin, yMax;
	xMin = [self pixelXFromDataX:NSMinX(dataRect)];
	yMin = [self pixelYFromDataY:NSMinY(dataRect)];
	xMax = [self pixelXFromDataX:NSMaxX(dataRect)];
	yMax = [self pixelYFromDataY:NSMaxY(dataRect)];
	return NSMakeRect(xMin, yMin, xMax-xMin, yMax-yMin);
}

// pixel space to data space

-(double)dataXFromPixelX:(double)pixelX;
{
	return NSMinX(_viewRect) + (pixelX - NSMinX(_plotRect))/NSWidth(_plotRect)*NSWidth(_viewRect);
}

-(double)dataYFromPixelY:(double)pixelY;
{
	double viewMinY = NSMinY(_viewRect);
	double viewMaxY = NSMaxY(_viewRect);
	if(_yAxisIsLogarithmic) {
		if(viewMinY < _logarithmicMinY)
			viewMinY = _logarithmicMinY;
		viewMinY = log(viewMinY);
		viewMaxY = log(viewMaxY);
	}
	double y = viewMinY + (pixelY - NSMinY(_plotRect)) / NSHeight(_plotRect) * (viewMaxY - viewMinY);
	if(_yAxisIsLogarithmic)
		y = exp(y);
	return y;
}

-(NSPoint)dataPointFromPixelPoint:(NSPoint)pixelPoint;
{
	return NSMakePoint([self dataXFromPixelX:pixelPoint.x],[self dataYFromPixelY:pixelPoint.y]);
}

-(NSRect)dataRectFromPixelRect:(NSRect)pixelRect;
{
	double xMin, xMax, yMin, yMax;
	xMin = [self dataXFromPixelX:NSMinX(pixelRect)];
	yMin = [self dataYFromPixelY:NSMinY(pixelRect)];
	xMax = [self dataXFromPixelX:NSMaxX(pixelRect)];
	yMax = [self dataYFromPixelY:NSMaxY(pixelRect)];
	return NSMakeRect(xMin, yMin, xMax-xMin, yMax-yMin);
}

// -----------------------
// miscellaneous functions
// -----------------------

- (NSColor*)colorForCurveAtIndex:(int)j
{
	NSColor *color=nil;
	if(colorDelegate != nil && [colorDelegate respondsToSelector:@selector(overrideColorForCurveAtIndex:)])
		color = [colorDelegate overrideColorForCurveAtIndex:j];
	if(!color)
		color = _plot[@"curves"][j][@"color"];
	return (color==nil ? [NSColor blackColor] : color);
}

- (void)resetCursorRects
{
	[super resetCursorRects];
	if(mouseDelegate != nil && [mouseDelegate respondsToSelector:@selector(cursorForPlotView)]) {
		NSCursor *cur = [mouseDelegate cursorForPlotView];
		[self addCursorRect:_plotRect cursor:cur];
		[cur setOnMouseEntered:YES];
	}
}

@end
