//
//  PlotView.h
//  PopBio
//
//  Created by Evan Jones on 4/2/07.
//  Last modified by Stanislav Kounitski on 8/16/07
//  This work is provided under the terms of the Educational Community License 1.0, a copy of which is included with the source code.
//

#import <Cocoa/Cocoa.h>

#define kHorizontalTextAreaSize 25.0
#define kVerticalTextAreaSize 15.0

// PLOT_MAX defines the largest number that we'll put
// on either axis in a plot.
#define PLOT_MAX 1e38

@interface PlotView: NSView{
	
	NSDictionary *_plot;
	NSRect _plotRect; // in Pixels -- Inset in  _labelRect.   The area where data can go
	NSRect _labelRect; // in Pixels -- Inset in [self frame].  The area where labels can go
	NSRect _xLabelRect;
	NSRect _yLabelRect;
	
	NSRect _dataRect;   // in data space
	NSRect _viewRect;
	NSRect _zoomRect;
	
	int _xIntervals;
	int _yIntervals;
	int _maxXLabels;
	int _maxYLabels;
	
	NSMutableArray *_xTicks; // in pixels
	NSMutableArray *_yTicks;
	NSMutableArray *_xLabels;
	NSMutableArray *_yLabels;
	
	BOOL _drawTicks;
	BOOL _yAxisIsLogarithmic;
	BOOL _drawZoomRect;
	BOOL _displayGrid;
	
	double _logarithmicMinY;
	
	NSMutableString *_plotTitle;
	NSMutableString *_yAxisLabel;
	NSMutableString *_xAxisLabel;
	
	// constants
	NSDictionary *_littleTextAttributes;
	NSDictionary *_bigTextAttributes;
	double _tickLength;
	
	IBOutlet id mouseDelegate;
	IBOutlet id colorDelegate;
	
	IBOutlet NSPanel *parametersPanel;
	IBOutlet NSTextField *minimumXTextField;
	IBOutlet NSTextField *maximumXTextField;
	IBOutlet NSTextField *xTicIncrementTextField;
	IBOutlet NSTextField *minimumYTextField;
	IBOutlet NSTextField *maximumYTextField;
	IBOutlet NSTextField *yTicIncrementTextField;
	IBOutlet NSTextField *plotTitleTextField;
	IBOutlet NSTextField *xAxisLabelTextField;
	IBOutlet NSTextField *yAxisLabelTextField;
	
	IBOutlet NSPopUpButton *saveMenuPopUpButton;
}

- (NSRect)boundingRectOfCurve:(NSDictionary *)curve;
- (NSRect)boundingRectOfAllCurves;
- (void)makePlotLabelsAgree;
- (void)makePlotValuesAgree;

- (void)setLabelCount;
- (void)setTicks;

- (double)pixelXFromDataX:(double)dataX;
- (double)pixelYFromDataY:(double)dataY;
- (NSPoint)pixelPointFromDataPoint:(NSPoint)dataPoint;
- (NSRect)pixelRectFromDataRect:(NSRect)dataRect;
- (double)dataXFromPixelX:(double)pixelX;
- (double)dataYFromPixelY:(double)pixelY;
- (NSPoint)dataPointFromPixelPoint:(NSPoint)pixelPoint;
- (NSRect)dataRectFromPixelRect:(NSRect)pixelRect;

- (NSColor*)colorForCurveAtIndex:(int)j;
- (NSString *)makeLabelForValue:(double)value withSpacing:(double)spacing;

- (NSString*)_plotTitle;
- (NSString*)_xAxisLabel;
- (NSString*)_yAxisLabel;

- (void)resetCursorRects;

- (void)setZoomRectEnabled:(BOOL)b;
- (void)setTicksEnabled:(BOOL)b;
- (void)setViewRect:(NSRect)viewRect;
- (void)setZoomRect:(NSRect)zoomRect;
- (NSRect)viewRect;
- (NSRect)dataRect;

- (void)dataRectHasChanged;

@end

@interface PlotView (PlotViewDrawing)

- (NSBezierPath*)axesPath;
- (NSBezierPath*)xTicksPath;
- (NSBezierPath*)yTicksPath;
- (NSBezierPath*)gridPath;
- (NSBezierPath*)pathForCurve:(NSDictionary*)curve;
- (NSBezierPath*)pathForLine:(NSDictionary*)line;
- (NSBezierPath*)pathForVerticalLine:(NSDictionary*)vLine;
- (NSBezierPath*)pathForParabola:(NSDictionary*)parabola;
- (NSBezierPath*)pathForHolling2Isocline:(NSDictionary*)h2iso;

- (void)writeHorizontalLabels;
- (void)writeVerticalLabels;
- (void)writeHeadings;

@end

@interface PlotView (PlotViewInteractivity)

- (IBAction)toggleDisplayGrid:(id)sender;
- (IBAction)toggleLogLinearYAxis:(id)sender;
- (IBAction)editDrawingParameters:(id)sender;
- (IBAction)closeParametersPanelOK:(id)sender;
- (IBAction)closeParametersPanelCancel:(id)sender;
- (IBAction)saveResultsAsText:(id)sender;
- (IBAction)saveParametersAsText:(id)sender;
- (IBAction)savePlotAsPDF:(id)sender;
- (IBAction)savePlotAsTIFF:(id)sender;

- (void)mouseDown:(NSEvent *)theEvent; 
- (void)mouseUp:(NSEvent *)theEvent; 
- (void)mouseDragged:(NSEvent *)theEvent;
- (void)addCurve:(NSDictionary*)aPlot;
- (void)removeCurve:(NSDictionary*)aPlot;
- (void)displayPlot:(NSDictionary*)aPlot;

@end

@interface NSObject (PlotViewMouseDelegate)

- (NSCursor*)cursorForPlotView;
- (void)mouseDownAtPoint:(NSPoint)pixelPoint;
- (void)mouseDraggedAtPoint:(NSPoint)pixelPoint;
- (void)mouseUpAtPoint:(NSPoint)pixelPoint;
- (void)dataRectHasChanged;

@end

@interface NSObject (PlotViewColorDelegate)

- (NSColor*)overrideColorForCurveAtIndex:(int)j;

@end