//
//  PopGrowthDocument.m
//  PopBio
//
//  Created by Aaron Golden on 05/09/07.
//  Copyright Reed College 2007 . All rights reserved.
//

#import "LifeTableDocument.h"
#import "PlotWindowController.h"
#import "PopBioController.h"
#import "AgeStructureView.h"
#import "SimpleTableView.h"

@implementation LifeTableDocument

// ----------------
// init and dealloc
// ----------------

- (id)init
{
    self = [super init];
    if (self) {
		timePeriods = 10;
		stochasticity = 0;
		inputTableStochasticity = 1;
		randomSeed = time(0);
		inputTableMatrixData = nil;
    }
    return self;
}

// --------------------------------------------------
// implementation of the document-specific parameters
// --------------------------------------------------

- (void)synchronizeParamsToUserInterface
{
	ageGroups = [inputTableParent numberOfRows];
	timePeriods = [timePeriodsField intValue];
	stochasticity = [stochasticButton state];
}

- (void)synchronizeUserInterfaceToParams
{
	[timePeriodsField setIntValue:timePeriods];
	[stochasticButton setState:stochasticity];
	
	if(inputTableStochasticity != stochasticity) {
	
		if(stochasticity) {
			[inputTableView addTableColumn:pxerrColumn];
			[inputTableView addTableColumn:mxerrColumn];
			[inputTableView moveColumn:4 toColumn:3];
		} else {
			[inputTableView removeTableColumn:pxerrColumn];
			[inputTableView removeTableColumn:mxerrColumn];
		}
		
		[inputTableView sizeToFit];
		inputTableStochasticity = stochasticity;
	}

	[inputTableView reloadData];
}

- (NSArray*)argumentsForEvaluator
{
	NSMutableArray *args = [NSMutableArray array];
	[args addObject:[NSString stringWithFormat:@"%d", kExperimentTypeLifeTable]];
	[args addObject:[NSString stringWithFormat:@"%d", ageGroups]];
	[args addObject:[NSString stringWithFormat:@"%d", timePeriods]];
	[args addObject:[NSString stringWithFormat:@"%d", stochasticity]];
	[args addObject:[NSString stringWithFormat:@"%d", randomSeed++]];
	NSString *tableString = [self dataAsString];
	[args addObjectsFromArray:[tableString componentsSeparatedByString:@" "]];
	return args;
}

- (NSString *)parametersReport;
{
	NSMutableString *report = [NSMutableString string];
	[report appendString:@"LIFE TABLE PARAMETERS REPORT\n----------------------------\n"];
	[report appendString:[NSString stringWithFormat:@"Time periods: %d\n", timePeriods]];
	[report appendString:[NSString stringWithFormat:@"Stochasticity: %@\n", stochasticity? @"On" : @"Off"]];
	[report appendString:[NSString stringWithFormat:@"Life table:\n%@\n", [self dataAsString]]];
	return report;
}

// ------------------------------------------------------
// document-specific implementation of setPlotCodeStrings
// ------------------------------------------------------

- (void)setPlotCodeStrings
{
	// create the fixedPlotCode string, if necessary
	if(!fixedPlotCode)
		fixedPlotCode = [NSString stringWithContentsOfFile:[[NSBundle bundleForClass:[self class]]
                                                             pathForResource:@"lifeTablePlotCode" ofType:@"txt"] encoding:NSUTF8StringEncoding error:NULL];
	
	generatedPlotCode = [[NSMutableString alloc] init];
	
	// generate generatedPlotCode
	[generatedPlotCode appendFormat:
		@"Params: VERBATIM\n%@\nEND VERBATIM\n\n",
		[self parametersReport]];

	[generatedPlotCode appendFormat:
		@"Results: VERBATIM\nLIFE TABLE RESULTS REPORT\n-------------------------\n%@\nEND VERBATIM\n\n",
		evaluatorOutput];
		
	[generatedPlotCode appendFormat:
		@"Curve ID 1: INT %d\nCurve ID 2: INT %d\n\n",
		[[PBController nextCurveID] intValue], [[PBController nextCurveID] intValue]];
	
	//NSLog(generatedPlotCode);
}

// -----------------------
// miscellaneous functions
// -----------------------

- (IBAction)helpButton:(id)sender
{
	[PBController openHelp:self];
}

- (void)windowControllerDidLoadNib:(NSWindowController *) aController
{
	[inputTableParent rememberColumns];
	[inputTableView sizeToFit];
	
	if(inputTableMatrixData) {
		[inputTableParent setData:inputTableMatrixData];
		inputTableMatrixData = nil;
	} else {
		[inputTableParent addRow:nil];
	}
	
	[self synchronizeUserInterfaceToParams];
}


- (NSString*)titleForPlotWindow
{
	return [NSString stringWithFormat:@"LifeTable Graphics: %@", [self displayName]];
}

- (NSString *)windowNibName
{
    return @"LifeTableDocument";
}

- (NSData *)dataOfType:(NSString *)typeName error:(NSError **)outError
{
	NSMutableData *data = [NSMutableData data];
	[data appendBytes:&timePeriods length:sizeof(int)];
	[data appendBytes:&stochasticity length:sizeof(int)];

	[data appendData:[inputTableParent getData]];

	return data;
}

- (BOOL)readFromData:(NSData *)data ofType:(NSString *)typeName error:(NSError **)outError
{
	memcpy(&timePeriods, [data bytes], sizeof(int));
	memcpy(&stochasticity, [data bytes]+sizeof(int), sizeof(int));
	
	inputTableMatrixData = [data subdataWithRange:
		NSMakeRange(sizeof(int)*2,[data length]-sizeof(int)*2)];

	return YES;
}

- (NSString *)dataAsString
{
	[self synchronizeParametersAndUserInterface:self];
	
	NSMutableArray *dataMatrix = [inputTableParent dataMatrix];
	NSMutableString *outputString = [NSMutableString string];

	int nx;
	double px, mx, pxerr, mxerr;

	int i;
	for(i=0; i<ageGroups; i++)
	{
		NSArray *row = dataMatrix[i];
		nx = [row[1] intValue];
		px = [row[2] doubleValue];
		mx = [row[4] doubleValue];
		
		if(!stochasticity)
			[outputString appendFormat:@"%d %.2lf %.2lf ", nx, px, mx];
		else
		{
			pxerr = [row[3] doubleValue];
			mxerr = [row[5] doubleValue];
			[outputString appendFormat:@"%d %.2f %.2f %.2f %.2f ", nx, px, pxerr, mx, mxerr];
		}
	}
	
	return [NSString stringWithString:outputString];
}

@end

@implementation LifeTableDocument(SimpleTableDataSourceDelegate)

- (void)didRemoveRowsFromTable:(NSTableView*)aTableView
{
	NSMutableArray *dataMatrix = [inputTableParent dataMatrix];
	int i, numRows = [dataMatrix count];
	for(i=0; i<numRows; i++)
		dataMatrix[i][0] = @(i);
}

- (NSMutableArray *)newRowForTable:(NSTableView*)aTableView
{
	NSMutableArray *dataMatrix = [inputTableParent dataMatrix];
	
	int x=0;
	if([dataMatrix count] > 0)
		x = [[dataMatrix lastObject][0] intValue] + 1;
	
	return [[NSMutableArray alloc] initWithObjects:
            @(x), @0,
            @1.0, @0.0,
            @0.0, @0.0, nil];
}

@end