/*
 *  LifeTable.c
 *  PopBio
 *
 *  Created by Aaron Golden on 7/20/07.
 *  This work is provided under the terms of the Educational Community License 1.0, a copy of which is included with the source code.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <math.h>
#include "LifeTable.h"
#include "BackEnd.h"

int runLifeTable(int argc, char *argv[])
{
	int numberOfRows, timePeriods, stochasticity, randomSeed;
	sscanf(argv[2], "%d", &numberOfRows);
	sscanf(argv[3], "%d", &timePeriods);
	sscanf(argv[4], "%d", &stochasticity);
	sscanf(argv[5], "%d", &randomSeed);

    if (numberOfRows <= 0) {
        fprintf(stderr, "PopBio (LifeTable) error, invalid number of rows parameter (%d).\n", numberOfRows);
        return -1;
    }

    int *n = (int*)malloc(sizeof(int)*(numberOfRows+1));
	double *p = (double*)malloc(sizeof(double)*(numberOfRows+1));
	double *perr = (double*)malloc(sizeof(double)*(numberOfRows+1));
	double *m = (double*)malloc(sizeof(double)*(numberOfRows+1));
	double *merr = (double*)malloc(sizeof(double)*(numberOfRows+1));
	double *plus_one = (double*)malloc(sizeof(double)*numberOfRows);
	double *q = (double*)malloc(sizeof(double)*numberOfRows);
	double *d = (double*)malloc(sizeof(double)*numberOfRows);
	double *e = (double*)malloc(sizeof(double)*numberOfRows);
	double *P = (double*)malloc(sizeof(double)*numberOfRows);
	double *Pn = (double*)malloc(sizeof(double)*numberOfRows);
	double *T = (double*)malloc(sizeof(double)*(timePeriods+1));
	if(!n || !p || !m || !perr || !merr || !plus_one || !q || !d || !e || !P || !Pn || !T){
		fprintf(stderr, "PopBio (LifeTable) error, memory allocation failed.\n");
        if (n) free(n);
        if (p) free(p);
        if (perr) free(perr);
        if (m) free(m);
        if (merr) free(merr);
        if (plus_one) free(plus_one);
        if (q) free(q);
        if (d) free(d);
        if (e) free(e);
        if (P) free(P);
        if (Pn) free(Pn);
        if (T) free(T);
		return -1;
	}

	printf("Number of Rows: INT %d\nTime Periods: INT %d\nStochasticity: INT %d\nIndex of oldest age group: INT %d\n\n",
		numberOfRows, timePeriods, stochasticity, numberOfRows-1);

	srandom(randomSeed);

	// Get the whole input table first.
	int row;
	for(row = 0; row < numberOfRows; row++){
		if(!stochasticity){
			sscanf(argv[6+3*row], "%d", &n[row]);
			sscanf(argv[6+3*row+1], "%lf", &p[row]);
			sscanf(argv[6+3*row+2], "%lf", &m[row]);
			perr[row] = 0.0;
			merr[row] = 0.0;
		}
		else{
			sscanf(argv[6+5*row], "%d", &n[row]);
			sscanf(argv[6+5*row+1], "%lf", &p[row]);
			sscanf(argv[6+5*row+2], "%lf", &perr[row]);
			sscanf(argv[6+5*row+3], "%lf", &m[row]);
			sscanf(argv[6+5*row+4], "%lf", &merr[row]);
		}
	}
	n[numberOfRows] = 0;
	p[numberOfRows] = 0.0;
	perr[numberOfRows] = 0.0;
	m[numberOfRows] = 0.0;
	merr[numberOfRows] = 0.0;
	
	// Then calculate the life table (easiest part)
	//
	// There is a confusing naming convention here.  When stochasticity is off
	// px refers to the probability that an individual survives from age x to age x+1,
	// l_{x+1}/l_{x}.  l_{x} refers to the probability that an individual survives
	// to age x, at all.
	//
	// On the other hand, when stochasticity is on, p_{x} refers to the probability that
	// an individual survives to enter age group x, that is l_{x}.  In our code we just
	// call the probability that an individual survives to age group x, p[x].  The
	// probability that an individual survives from x to x+1 is plus_one[x].
	double pxmxTotal=0.0;
	
	printf("Life Table Matrix: MATRIX\n");
	for(row = 0; row < numberOfRows; row++){
		// Calculating almost everything in the life table is a one-line per column affair.
		if(p[row] > 0.0)
			plus_one[row] = p[row+1]/p[row];
		else
			plus_one[row] = 0.0;
			
		q[row] = 1.0 - plus_one[row];
		d[row] = p[row]-p[row+1];

		// Calculating e[x] is a little more complicated
		if(p[row] > 0.0){
			e[row] = 0.0;
			int j;
			for(j = row+1; j < numberOfRows; j++)
				e[row] += p[j];
			e[row] /= p[row];
		}
		else
			e[row] = 0.0;
			
		printf("%d\t%.2lf\t%.2lf\t%.2lf\t%.2lf\t%.2lf\t%.2lf\t%.2lf\n",
			row, plus_one[row], q[row], d[row], e[row], p[row], m[row], p[row]*m[row]);
		
		pxmxTotal += p[row]*m[row];
	}
	printf("Total\t \t \t \t \t \t \t%.2lf\n", pxmxTotal);
	printf("END MATRIX\n\n");

	// ------------------------------------------------------

	// Compute estimated time between generations and rate of growth.
	// These computations should appear in the comment field of the
	// total population curve.
	
	double R_0 = 0.0;
	double T_between_gens = 0.0;
	double l_0;
	
	for(row = 0; row < numberOfRows; row++) {
		double lx = p[row];
		double mx = m[row];
		R_0 += lx*mx;
		T_between_gens += lx*mx*(double)row;
		if(row==0)l_0 = lx;
	}
	
	R_0 /= l_0;
	T_between_gens /= R_0;
	double r_estimate = log(R_0)/T_between_gens;
	printf("Total Population Comment: T=%lf, r ~ %lf\n\n", T_between_gens, r_estimate);
	
	// ------------------------------------------------------

	printf("BEGIN DATA\n");
	//printf("Age\tln(l(x))\n");
	printf("Survivorship X\tSurvivorship Y\n");
	int x;
	for(x = 0; x < numberOfRows; x++)
		printf("%d\t%lf\n", x, 1000.0*p[x]);
	printf("END DATA\n\n");
	
	// Now calculate the age structure at each time:
	//
	// P(x,t) = P(x-1,t-1)*plus_one(x-1), for x > 0;
	// P(0,t) = \sum_{x=0}^{k-1}{P(x,t)*m(x)}
	//
	// Note that we need to double buffer (P and Pn here).
	// We *could* get away with modifying P in place by starting
	// at x = numberOfRows-1 and working backwards in the computation
	// of P(x,t) at each step, but it would make the method overly
	// subtle for a non-noticable speed improvement.
	//
	// We calculate the total population curve at the same time
	// as the age structure data, and store that in T.
	
	// The age structure result is the only part that should be affected
	// by stochasticity being turned on.
	printf("Age Structure Matrix: MATRIX\n");
	for(x = 0; x < numberOfRows; x++)
		P[x] = (double)n[x];

	for(int t = 0; t <= timePeriods; t++){
		for(x = 0; x < numberOfRows; x++)
			printf("%.2lf\t", P[x]);
		
		// Get the total population at the start
		// of this time interval.
		T[t] = 0.0;
		for(x = 0; x < numberOfRows; x++)
			T[t] += P[x];
			
		printf("%.2lf\n", T[t]);
		
		for(x = 1; x < numberOfRows; x++)
			Pn[x] = P[x-1]*(plus_one[x-1]+_randomDouble()*perr[x-1]-_randomDouble()*perr[x-1]);
		Pn[0] = 0.0;
		for(x = 0; x < numberOfRows; x++)
			Pn[0] += P[x]*(m[x]+_randomDouble()*merr[x]-_randomDouble()*merr[x]);
		memcpy(P, Pn, sizeof(double)*numberOfRows);
	}
	printf("END MATRIX\n\n");
	
	// Print out the Total Population curve.
	printf("BEGIN DATA\n");
	printf("Time Intervals\tTotal Population Size\n");
	for(int t = 0; t <= timePeriods; t++)
		printf("%d\t%lf\n", t, T[t]);
	printf("END DATA\n");
	
	// Free up memory
	free(P);
	free(Pn);
	free(T);
	free(plus_one);
	free(q);
	free(d);
	free(e);
	free(n);
	free(p);
	free(perr);
	free(m);
	free(merr);
	
	return 0;
}