//
//  CompetitionDocument.m
//  PopBio
//
//  Created by Aaron Golden on 05/09/07.
//  Copyright Reed College 2007 . All rights reserved.
//

#import "CompetitionDocument.h"
#import "PlotWindowController.h"
#import "PopBioController.h"
#import "FineTunePanel.h"
#import "NSColorAdditions.h"
#import "NSDictionaryCurve.h"

@implementation CompetitionDocument

// ----------------
// init and dealloc
// ----------------

- (id)init
{
	self = [super init];
	if(self){
		NSString *str = [[NSBundle bundleForClass:[self class]] pathForImageResource:@"targetCursor.png"];
		_crosshairImage = [[NSImage alloc] initWithContentsOfFile:str];
		
		params.timeIntervals = 50.0;
		params.initialPopulation1 = 20.0;
		params.initialPopulation2 = 20.0;
		params.rateOfIncrease1 = 0.5;
		params.rateOfIncrease2 = 0.25;
		params.carryingCapacity1 = 100.0;
		params.carryingCapacity2 = 100.0;
		params.competitionCoefficient1 = 0.5;
		params.competitionCoefficient2 = 0.75;
		params.integratorType = IntegratorTypeRungeKutta;
		params.pointStep = 0.1;
		params.randomSeed = time(0);
	}
	return self;
}


// --------------------------------------------------
// implementation of the document-specific parameters
// --------------------------------------------------

- (void)synchronizeParamsToUserInterface
{
	params.timeIntervals = [timeIntervalsTextField doubleValue];
	params.initialPopulation1 = [initialPopulation1TextField doubleValue];
	params.rateOfIncrease1 = [rateOfIncrease1TextField doubleValue];
	params.competitionCoefficient1 = [competitionCoefficient1TextField doubleValue];
	params.carryingCapacity1 = [carryingCapacity1TextField doubleValue];
	params.initialPopulation2 = [initialPopulation2TextField doubleValue];
	params.rateOfIncrease2 = [rateOfIncrease2TextField doubleValue];
	params.competitionCoefficient2 = [competitionCoefficient2TextField doubleValue];
	params.carryingCapacity2 = [carryingCapacity2TextField doubleValue];
	params.integratorType = [fineTunePanel integratorType];
	params.pointStep = [fineTunePanel pointStep];
}

- (void)synchronizeUserInterfaceToParams
{
	[timeIntervalsTextField setDoubleValue:params.timeIntervals];
	[initialPopulation1TextField setDoubleValue:params.initialPopulation1];
	[rateOfIncrease1TextField setDoubleValue:params.rateOfIncrease1];
	[competitionCoefficient1TextField setDoubleValue:params.competitionCoefficient1];
	[carryingCapacity1TextField setDoubleValue:params.carryingCapacity1];
	[initialPopulation2TextField setDoubleValue:params.initialPopulation2];
	[rateOfIncrease2TextField setDoubleValue:params.rateOfIncrease2];
	[competitionCoefficient2TextField setDoubleValue:params.competitionCoefficient2];
	[carryingCapacity2TextField setDoubleValue:params.carryingCapacity2];
	[fineTunePanel setIntegratorType:params.integratorType];
	[fineTunePanel setPointStep:params.pointStep];
}

- (NSArray*)argumentsForEvaluator
{
	return @[[NSString stringWithFormat:@"%d", kExperimentTypeCompetition],
		[NSString stringWithFormat:@"%lf", params.timeIntervals],
		[NSString stringWithFormat:@"%lf", params.initialPopulation1],
		[NSString stringWithFormat:@"%lf", params.rateOfIncrease1],
		[NSString stringWithFormat:@"%lf", params.competitionCoefficient1],
		[NSString stringWithFormat:@"%lf", params.carryingCapacity1],
		[NSString stringWithFormat:@"%lf", params.initialPopulation2],
		[NSString stringWithFormat:@"%lf", params.rateOfIncrease2],
		[NSString stringWithFormat:@"%lf", params.competitionCoefficient2],
		[NSString stringWithFormat:@"%lf", params.carryingCapacity2],
		[NSString stringWithFormat:@"%.8f", [[[NSUserDefaults standardUserDefaults]
			objectForKey:@"competitionStabilityThreshold"] doubleValue]],
		[NSString stringWithFormat:@"%d", params.integratorType],
		[NSString stringWithFormat:@"%lf", params.pointStep],
		[NSString stringWithFormat:@"%d", params.randomSeed++]];
}

- (NSString *)parametersReport;
{
	NSMutableString *str = [NSMutableString string];
	[str appendFormat:@"Competition:\n\n"];
	[str appendFormat:@"Time Intervals: %.1lf\n",params.timeIntervals];
	[str appendFormat:@"Species 1:\n"];
	[str appendFormat:@"  Initial Population Size: %.1lf\n",params.initialPopulation1];
	[str appendFormat:@"  Rate of Increase: %.3lf\n",params.rateOfIncrease1];
	[str appendFormat:@"  Carrying Capacity: %d\n",(int)(params.carryingCapacity1)];
	[str appendFormat:@"  Competition Coefficient: %.3lf\n",params.competitionCoefficient1];
	[str appendFormat:@"Species 2:\n"];
	[str appendFormat:@"  Initial Population Size: %.1lf\n",params.initialPopulation2];
	[str appendFormat:@"  Rate of Increase: %.3f\n",params.rateOfIncrease2];
	[str appendFormat:@"  Carrying Capacity: %d\n",(int)(params.carryingCapacity2)];
	[str appendFormat:@"  Competition Coefficient: %.3lf\n",params.competitionCoefficient2];
	[str appendFormat:@"Epsilon: %.12lf\n",[[[NSUserDefaults standardUserDefaults]
		objectForKey:@"competitionStabilityThreshold"] doubleValue]];	
	if(params.integratorType == IntegratorTypeRungeKutta)
		[str appendFormat:@"Runge-Kutta Integration\n"];
	else if(params.integratorType == IntegratorTypeEuler)
		[str appendFormat:@"Euler forward Integration\n"];
	[str appendFormat:@"Point step: %lf",params.pointStep];
	return str;
}

// ------------------------------------------------------
// document-specific implementation of setPlotCodeStrings
// ------------------------------------------------------

- (void)setPlotCodeStrings
{
	// create the fixedPlotCode string, if necessary
	if(!fixedPlotCode)
		fixedPlotCode = [NSString stringWithContentsOfFile:[[NSBundle bundleForClass:[self class]]
                                                             pathForResource:@"competitionPlotCode" ofType:@"txt"] encoding:NSUTF8StringEncoding error:NULL];
	
	generatedPlotCode = [[NSMutableString alloc] init];
	
	// generate generatedPlotCode
	[generatedPlotCode appendFormat:
		@"Params: VERBATIM\n%@\nEND VERBATIM\n\n",
		[self parametersReport]];
	
	[generatedPlotCode appendFormat:
		@"Color 1: CLR %@\nColor 2: CLR %@\n\n",
		[[colorWell1 color] stringValue], [[colorWell2 color] stringValue]];
	
	[generatedPlotCode appendFormat:
		@"Curve ID 1: INT %d\nCurve ID 2: INT %d\nCurve ID 3: INT %d\nCurve ID 4: INT %d\nCurve ID 5: INT %d\n\n",
		[[PBController nextCurveID] intValue], [[PBController nextCurveID] intValue], [[PBController nextCurveID] intValue], [[PBController nextCurveID] intValue], [[PBController nextCurveID] intValue]];
	
	//NSLog(generatedPlotCode);
}

// -----------------------
// miscellaneous functions
// -----------------------

- (IBAction)helpButton:(id)sender
{
	[PBController openHelp:self];
}

- (NSString*)titleForPlotWindow
{
	return [NSString stringWithFormat:@"Competition Graphics: %@", [self displayName]];
}

- (void)registerForNotifications
{
	[[NSNotificationCenter defaultCenter] addObserver:self
		selector:@selector(controlTextDidEndEditing:)
		name:@"NSControlTextDidEndEditingNotification" object:nil];
}

- (NSString *)windowNibName
{
    return @"CompetitionDocument";
}

- (NSData *)dataOfType:(NSString *)typeName error:(NSError **)outError
{
	[self synchronizeParamsToUserInterface];
	return [NSData dataWithBytes:&params length:sizeof(CompetitionParameters)];
}

- (BOOL)readFromData:(NSData *)data ofType:(NSString *)typeName error:(NSError **)outError
{
	memcpy(&params, [data bytes], sizeof(CompetitionParameters));
	[self synchronizeUserInterfaceToParams];
    return YES;
}

- (IBAction)toggleInitialPopulationPanelDrawer:(id)sender
{
	if([initialPopulationPanelDrawer state] == NSDrawerClosedState) {
		[initialPopulationPanelDrawer open];
		[initialPopulationPanelButton setTitle:@"Hide Initial Population Selector"];
		[initialPopulationPanelButton setNeedsDisplay:YES];
	}
	else {
		[initialPopulationPanelDrawer close];
		[initialPopulationPanelButton setTitle:@"Show Initial Population Selector"];
		[initialPopulationPanelButton setNeedsDisplay:YES];
	}
}

- (void)updateInitialPopulationPanelPlotView
{
	double x = [initialPopulation1TextField doubleValue];
	double y = [initialPopulation2TextField doubleValue];
	double K1 = [carryingCapacity1TextField doubleValue];
	double K2 = [carryingCapacity2TextField doubleValue];
	double a21 = [competitionCoefficient1TextField doubleValue];
	double a12 = [competitionCoefficient2TextField doubleValue];
	NSColor *color1 = [colorWell1 color];
	NSColor *color2 = [colorWell2 color];

	NSDictionary *isocline1 = [NSDictionary lineWithSlope:(-1/a12) intercept:(K1/a12) color:color1];
	NSDictionary *isocline2 = [NSDictionary lineWithSlope:(-a21)   intercept:(K2)     color:color2];
	NSDictionary *crosshair = [NSDictionary imageWithImage:_crosshairImage X:x Y:y xOffset:8 yOffset:8 opacity:0.5];
	
	NSDictionary *plot = @{@"long-title": @"Initial Population Selector",
		@"x-axis-label": @"Species 1 Population",
		@"y-axis-label": @"Species 2 Population",
		@"x-min": @0,
		@"y-min": @0,
		@"x-max": @(K1),
		@"y-max": @(K2),
		@"x-ticks": @5,
		@"y-ticks": @5,
		@"curves": @[isocline1, isocline2, crosshair]};
	
	[initialPopulationPanelPlotView displayPlot:plot];
}

- (void)controlTextDidEndEditing:(NSNotification *)aNotification
{
	[self updateInitialPopulationPanelPlotView];
}

- (void)mouseDownAtDataPoint:(NSPoint)dataPoint
{
	[initialPopulation1TextField setDoubleValue:dataPoint.x];
	[initialPopulation2TextField setDoubleValue:dataPoint.y];
	[self updateInitialPopulationPanelPlotView];
}

@end

@implementation CompetitionDocument(NSNibAwaking)

-(void)awakeFromNib
{
	[self registerForNotifications];
	[self updateInitialPopulationPanelPlotView];
}
@end

@implementation CompetitionDocument(NSDrawerDelegate)

- (void)drawerDidClose:(NSNotification *)notification
{
	[initialPopulationPanelButton setTitle:@"Show Initial Population Selector"];
	[initialPopulationPanelButton setNeedsDisplay:YES];
}

@end