/*
 *  Competition.c
 *  PopBio
 *
 *  Created by Aaron Golden on 05/09/07.
 *  This work is provided under the terms of the Educational Community License 1.0, a copy of which is included with the source code.
 *
 */
 
#include <stdio.h>
#include <math.h>
#include <stdlib.h>

#include "Backend.h"
#include "Competition.h"
#include "RungeKutta.h"

double r1, r2, a12, a21, K1, K2;

pair xPrimeForLotkaVolterra(double t, pair x)
{
	pair xPrime;
	xPrime.a = r1*x.a*(1.0 - (x.a + a12*x.b)/K1);
	xPrime.b = r2*x.b*(1.0 - (x.b + a21*x.a)/K2);
	return xPrime;
}

int runCompetition(int argc, char *argv[])
{
	if(argc < 15){
		printf("Competition requires parameters:  <timeIntervals> <initial population size 1>\
		<rate of increase 1> <competition coefficient 1> <carrying capacity 1>\
		<initial population size 2> <rate of increase 2> <competition coefficient 2>\
		<carrying capacity 2> <epsilon>	<integrator type> <point step> <random seed>\n");
		return -1;
	}

	CompetitionParameters params;
	sscanf(argv[2], "%lf", &params.timeIntervals);
	sscanf(argv[3], "%lf", &params.initialPopulation1);
	sscanf(argv[4], "%lf", &params.rateOfIncrease1);
	sscanf(argv[5], "%lf", &params.competitionCoefficient1);
	sscanf(argv[6], "%lf", &params.carryingCapacity1);
	sscanf(argv[7], "%lf", &params.initialPopulation2);
	sscanf(argv[8], "%lf", &params.rateOfIncrease2);
	sscanf(argv[9], "%lf", &params.competitionCoefficient2);
	sscanf(argv[10], "%lf", &params.carryingCapacity2);
	sscanf(argv[11], "%lf", &params.epsilon);
	sscanf(argv[12], "%d", &params.integratorType);
	sscanf(argv[13], "%lf", &params.pointStep);
	sscanf(argv[14], "%d", &params.randomSeed);
	
	srandom(params.randomSeed);
	
	int numTimeSteps = (int)ceil(params.timeIntervals/params.pointStep)+1;
	pair x;
	x.a = params.initialPopulation1;
	x.b = params.initialPopulation2;
	
	r1 = params.rateOfIncrease1;
	r2 = params.rateOfIncrease2;
	a21 = params.competitionCoefficient1;
	a12 = params.competitionCoefficient2;
	K1 = params.carryingCapacity1;
	K2 = params.carryingCapacity2;
	double dt = params.pointStep;
	
	double *x1Table = (double*)malloc(sizeof(double)*numTimeSteps);
	double *x2Table = (double*)malloc(sizeof(double)*numTimeSteps);
	x1Table[0] = x.a;
	x2Table[0] = x.b;

	double t=0.0;
	int j;
	int stabilized = 0;
	
	printf("Isocline 1 Slope: DBL %lf\n", -1/a12);
	printf("Isocline 1 Intercept: DBL %lf\n", K1/a12);
	printf("Isocline 2 Slope: DBL %lf\n", -a21);
	printf("Isocline 2 Intercept: DBL %lf\n\n", K2);
	
	printf("Species 1 Population Max: INT %d\n", (int)ceil(K1));
	printf("Species 2 Population Max: INT %d\n", (int)ceil(K2));
	printf("Population Max: INT %d\n\n", (int)ceil((K1 > K2 ? K1 : K2)));
	
	printf("BEGIN DATA\n");
	printf("Time\tSpecies 1\tSpecies 2\n");
	printf("%lf\t\%lf\t\%lf\n", t, x.a, x.b);

	for(j=1; j < numTimeSteps; j++) {
		t += dt;

		if(params.integratorType == IntegratorTypeEuler) {
			x.a += (r1*x.a*(1.0 - (x.a + a12*x.b)/K1)) * dt;
			x.b += (r2*x.b*(1.0 - (x.b + a21*x.a)/K2)) * dt;
		}
		
		else if(params.integratorType == IntegratorTypeRungeKutta)
			x = getNextRungeKuttaPairValue(x, t, &xPrimeForLotkaVolterra, dt);
		
		// Clamp the population minimum size at zero
		if(x.a < 0.0) x.a = 0.0;
		if(x.b < 0.0) x.b = 0.0;
		// Clamp the population maximum size at 1e100 just like the original PopBio did.
		if(x.a > 1e100) x.a = 1e100;
		if(x.b > 1e100) x.b = 1e100;
		
		x1Table[j] = x.a;
		x2Table[j] = x.b;
		
		printf("%lf\t\%lf\t\%lf\n", t, x.a, x.b);
		
		if( fabs(x1Table[j]-x1Table[j-1]) / dt < params.epsilon &&
		    fabs(x2Table[j]-x2Table[j-1]) / dt < params.epsilon ) {
			stabilized = 1;
			break;
		}
	}
	
	printf("END DATA\n");
	
	if(stabilized)
		printf("\ncomment: Stabilized after %lf time periods\n", t);
	
	free(x1Table);
	free(x2Table);
	
	return 0;
}