//
//  AgeStructureView.m
//  PopBio
//
//  Created by Aaron Golden on 8/1/07.
//  This work is provided under the terms of the Educational Community License 1.0, a copy of which is included with the source code.
//

#import "AgeStructureView.h"

@implementation AgeStructureView

- (id)initWithFrame:(NSRect)frame {
    self = [super initWithFrame:frame];
    if (self) {
		parametersString = [[NSMutableString alloc] init];
		resultsString = [[NSMutableString alloc] init];
    }
    return self;
}

- (IBAction)saveParametersAsText:(id)sender
{
	NSString *defaultFilename = @"Untitled Parameters";
	NSSavePanel *sp = [NSSavePanel savePanel];
    [sp setAllowedFileTypes:@[@"txt"]];
    [sp setDirectoryURL:nil];
    [sp setNameFieldStringValue:defaultFilename];
    int runResult = [sp runModal];
	if (runResult == NSOKButton) {
        NSError *writeError = nil;
		if (![parametersString writeToURL:[sp URL] atomically:YES encoding:NSUTF8StringEncoding error:&writeError]){
			[[NSAlert alertWithMessageText:@"An error occurred while saving."
				defaultButton:@"OK" alternateButton:nil otherButton:nil
				informativeTextWithFormat:@"An error occurred while attempting to save the parameters: %@", [writeError localizedDescription]]
				beginSheetModalForWindow:[self window] modalDelegate:nil didEndSelector:nil contextInfo:nil];
		}
	}
	
	[popUpButton selectItemAtIndex:0];
}

- (IBAction)saveResultsAsText:(id)sender
{
	NSString *defaultFilename = @"Untitled Results";
	NSSavePanel *sp = [NSSavePanel savePanel];
    [sp setAllowedFileTypes:@[@"txt"]];
    [sp setDirectoryURL:nil];
    [sp setNameFieldStringValue:defaultFilename];
    int runResult = [sp runModal];
	if (runResult == NSOKButton) {
        NSError *writeError = nil;
		if (![resultsString writeToURL:[sp URL] atomically:YES encoding:NSUTF8StringEncoding error:&writeError]){
			[[NSAlert alertWithMessageText:@"An error occurred while saving."
				defaultButton:@"OK" alternateButton:nil otherButton:nil
				informativeTextWithFormat:@"An error occurred while attempting to save the results: %@", [writeError localizedDescription]]
				beginSheetModalForWindow:[self window] modalDelegate:nil didEndSelector:nil contextInfo:nil];
		}
	}
	
	[popUpButton selectItemAtIndex:0];
}

- (void)synchronizeAgeStructureResultsControlsToTimeElapsed
{
	[slider setDoubleValue:time];
	[stepper setIntValue:(int)time];
	[timePeriodsElapsedLabel setStringValue:
		[NSString stringWithFormat:@"%d time period%s elapsed", (int)time, ((int)time!=1)?"s":"" ]];
	[slider setNeedsDisplay:YES];
	[tableView setNeedsDisplay:YES];
	[timePeriodsElapsedLabel setNeedsDisplay:YES];
}

- (void)setResultsMatrix:(NSArray*)aMatrix
{
    if ([aMatrix isKindOfClass:[NSArray class]]) {
        resultsMatrix = [aMatrix mutableCopy];
        [slider setEnabled:YES];
        [slider setMaxValue:(double)([resultsMatrix count]-1)];
        [stepper setEnabled:YES];
        [stepper setMaxValue:(double)([resultsMatrix count]-1)];
        [playPauseButton setEnabled:YES];
    } else {
        resultsMatrix = nil;
        [slider setEnabled:NO];
        [slider setMaxValue:0];
        [stepper setEnabled:NO];
        [stepper setMaxValue:0];
        [playPauseButton setEnabled:NO];
    }
}

- (void)setParametersString:(NSString*)aString
{
	[parametersString setString:aString];
}

- (void)setResultsString:(NSString*)aString
{
	[resultsString setString:aString];
}

- (void)setPopUpButton:(NSPopUpButton*)aPopUpButton
{
	popUpButton = aPopUpButton;
}

BOOL doPlayAgeStructureResultsShouldDie;
- (void)doPlayAgeStructureResults:(id)object
{
	@autoreleasepool {
	
		double maxTime = (double)([resultsMatrix count]-1);
		double timeIncrement = 1.0 / 60.0, barIncrement = maxTime / 600.0; /* Let's do 60 frames a second for 10 seconds */
		
		if(time == maxTime)
			time = 0.0;

		while(time <= maxTime-barIncrement && doPlayAgeStructureResultsShouldDie == NO) {
			[self synchronizeAgeStructureResultsControlsToTimeElapsed];
			time += barIncrement;
			[NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:timeIncrement]];		
		}
		if(time >= maxTime - barIncrement)
			time = maxTime;
		[self synchronizeAgeStructureResultsControlsToTimeElapsed];
			
		[self pauseAgeStructureResults:nil];
	}
}

- (IBAction)setTimeInAgeStructureResults:(id)sender
{
	[self pauseAgeStructureResults:sender];
	time = [slider doubleValue];
	[self synchronizeAgeStructureResultsControlsToTimeElapsed];
}

- (IBAction)stepThroughAgeStructureResults:(id)sender
{
	[self pauseAgeStructureResults:sender];
	time = (double)[stepper intValue];
	[self synchronizeAgeStructureResultsControlsToTimeElapsed];
}

- (IBAction)playAgeStructureResults:(id)sender
{
		
	doPlayAgeStructureResultsShouldDie = NO;
	[NSThread detachNewThreadSelector:@selector(doPlayAgeStructureResults:)
		toTarget:self withObject:nil];
		
	[playPauseButton setAction:@selector(pauseAgeStructureResults:)];
	[playPauseButton setTitle:@"Pause"];	
}

- (IBAction)pauseAgeStructureResults:(id)sender
{
	doPlayAgeStructureResultsShouldDie = YES;

	[playPauseButton setAction:@selector(playAgeStructureResults:)];
	[playPauseButton setTitle:@"Play"];	
}

- (double)populationInAgeGroup:(int)ageGroup atTime:(double)t
{
	int low_index = (int)floor(t);
	int high_index = (int)ceil(t);
	
	if(high_index == low_index)
		return [resultsMatrix[low_index][ageGroup] doubleValue];
	
	double lowPoint = [resultsMatrix[low_index][ageGroup] doubleValue];
	double highPoint = [resultsMatrix[high_index][ageGroup] doubleValue];
	return lowPoint + (time-low_index)*(highPoint-lowPoint);
}

- (double)totalPopulationAtTime:(double)t
{
	return [self populationInAgeGroup:[self numberOfRowsInTableView:nil] atTime:t];
}

@end

@implementation AgeStructureView(NSTableDataSource)

- (NSInteger)numberOfRowsInTableView:(NSTableView *)aTableView
{
	return [resultsMatrix[0] count]-1;
}

- (NSCell *)tableView:(NSTableView *)tableView dataCellForTableColumn:(NSTableColumn *)tableColumn row:(NSInteger)row {
    if([[tableColumn identifier] intValue] == 3) {
		double T = [self totalPopulationAtTime:time];
		if(T > 0) {
            NSLevelIndicatorCell *levelIndicatorCell = [[NSLevelIndicatorCell alloc] initWithLevelIndicatorStyle:NSRelevancyLevelIndicatorStyle];
            [levelIndicatorCell setMinValue:0.0];
            [levelIndicatorCell setMaxValue:1.0];
            [levelIndicatorCell setDoubleValue:0];
            return levelIndicatorCell;
		}
	}

	return nil;
}

- (id)tableView:(NSTableView *)aTableView objectValueForTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex {
	int columnIndex = [[aTableColumn identifier] intValue];

	// The first column is just the age group.
	if(columnIndex == 0)
		return [NSString stringWithFormat:@"%d", rowIndex];
	
	// The second column is the current population
	else if(columnIndex == 1)
		return @([self populationInAgeGroup:rowIndex atTime:time]);
	
	// The third column contains the population of this age group
	// as a percentage of the total population at the current time
	if(columnIndex == 2) {
		double p = [self populationInAgeGroup:rowIndex atTime:time];
		double T = [self totalPopulationAtTime:time];
		
		if(T == 0.0)
			return @"Undefined";
		else
			return [NSString stringWithFormat:@"%.1lf%%", p/T*100];
	}
	
	// the fourth column contains the same ratio in a progress bar
	else if(columnIndex == 3) {
        double p = [self populationInAgeGroup:rowIndex atTime:time];
        double T = [self totalPopulationAtTime:time];
        if (T > 0) {
            return @(p / T);
        } else {
			return @"Undefined (total population is zero)";
        }
	}
	
	// the last column contains the (discrete) rate of change of this
	// age group from the previous time period to the current one
	else if(columnIndex==4) {
		int period = (int)floor(time);
		if(period == 0)
			return @"-";
		double current = [resultsMatrix[period][rowIndex] doubleValue];
		double prev = [resultsMatrix[period-1][rowIndex] doubleValue];
		if(prev == 0)
			return @"-";
		else
			return [NSString stringWithFormat:@"%.2lf", current/prev];
	}
	
	return nil;
}

@end
