//
//  AccumulatorWindowController.m
//  PopBio
//
//  Created by Aaron Golden on 4/3/07.
//  This work is provided under the terms of the Educational Community License 1.0, a copy of which is included with the source code.
//

#import "AccumulatorWindowController.h"
#import "PlotView.h"
#import "NSDictionaryCurve.h"

@implementation AccumulatorWindowController

- (id)initWithWindow:(NSWindow *)window
{
	self = [super initWithWindow:window];
	if(self)
		_curves = [[NSMutableArray alloc] init];
	return self;
}

- (IBAction)toggleShowPlotTable:(id)sender
{
	if([plotDrawer state] == NSDrawerOpenState)
		[plotShowHideButton setTitle:@"Show Curves List"];
	else if([plotDrawer state] == NSDrawerClosedState)
		[plotShowHideButton setTitle:@"Hide Curves List"];
	
	[plotDrawer toggle:sender];
	[plotShowHideButton setNeedsDisplay:YES];
}

- (void)setPlot:(NSDictionary*)aPlot
{
	NSMutableDictionary *plotCopy = [NSMutableDictionary dictionaryWithDictionary:aPlot];
	[plotCopy removeObjectForKey:@"curves"];
	plotCopy[@"curves"] = [NSMutableArray array];
	_plot = plotCopy;
	[plotView displayPlot:plotCopy];
}

- (BOOL)hasPlot
{
	return !(_plot == nil);
}

- (void)addCurve:(NSDictionary*)aCurve withName:(NSString*)aName
{
	NSMutableDictionary *curveCopy = [[NSMutableDictionary alloc] initWithDictionary:aCurve];
	[_curves addObject:@{@"curve": curveCopy, @"name": aName}];
	[plotTable reloadData];
	[plotView addCurve:curveCopy];
}

- (void)removeCurve:(NSDictionary*)aCurve
{
	int j;
	for(j=0; j < [_curves count]; j++){
		if(_curves[j][@"curve"] == aCurve){
			[_curves removeObjectAtIndex:j];
			break;
		}
	}
	[plotTable reloadData];	
	[plotView removeCurve:aCurve];
}

- (BOOL)hasCurve:(NSDictionary*)aCurve
{
	int j;
	for(j=0; j < [_curves count]; j++) {
		if([_curves[j][@"curve"] isEqualToCurve:aCurve])
			return YES;
	}
	return NO;
}

- (NSInteger)numberOfRowsInTableView:(NSTableView *)aTableView
{
	return [_curves count];
}

- (id)tableView:(NSTableView *)aTableView objectValueForTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
	return _curves[rowIndex][@"name"];
}

- (void)tableView:(NSTableView *)aTableView setObjectValue:(id)anObject forTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
}

- (IBAction)removeSelectedCurves:(id)sender
{
	NSIndexSet *selectedRowIndexes = [plotTable selectedRowIndexes];
	NSArray *curvesToRemove = [_curves objectsAtIndexes:selectedRowIndexes];
	int j;
	for(j = 0; j < [curvesToRemove count]; j++){
		[plotView removeCurve:curvesToRemove[j][@"curve"]];
	}
	[_curves removeObjectsAtIndexes:selectedRowIndexes];
	[plotView setNeedsDisplay:YES];
	[plotTable reloadData];
}

- (IBAction)updatePlotHiliting:(id)sender
{
	[plotView setNeedsDisplay:YES];
}

@end

@implementation AccumulatorWindowController (PlotViewColorDelegate)

- (NSColor*)overrideColorForCurveAtIndex:(int)j
{
	if([plotTable isRowSelected:j])
		return [NSColor redColor];
	return nil;
}

@end

@implementation AccumulatorWindowController(NSWindowDelegate)
- (void)clearTheAccumulator
{
	int j;
	for(j = 0; j < [_curves count]; j++){
		[plotView removeCurve:_curves[j][@"curve"]];
	}
	[_curves removeAllObjects];
	[plotTable reloadData];
}

- (void)accumulatorCloseVerificationDidEnd:(NSWindow *)sheet returnCode:(int)returnCode contextInfo:(void*)contextInfo
{
	if(returnCode == NSAlertAlternateReturn){
		[self clearTheAccumulator];
		[[self window] close];
	}
	
	[sheet orderOut:self];
}

- (BOOL)windowShouldClose:(id)sender
{
	BOOL verifyClosingAccumulator = [[NSUserDefaults standardUserDefaults] boolForKey:@"verifyClosingAccumulator"];
	BOOL willClose = YES;
	if(verifyClosingAccumulator){
		NSBeginAlertSheet(@"Are you sure that you want to close the accumulator window?",
			@"Don't close", @"Close", nil, [self window], self, @selector(accumulatorCloseVerificationDidEnd:returnCode:contextInfo:),
			nil, nil, @"Closing the accumulator will remove all plots from the accumulator.");
		willClose = NO;
	}
	else{
		[self clearTheAccumulator];
	}
	
	return willClose;
}

@end
